<?php
// Webhook endpoint untuk menerima update dari Telegram

// Disable error reporting for production
error_reporting(0);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// Set header untuk JSON response
header('Content-Type: application/json');

try {
    // Check if required files exist
    if (!file_exists('handlers/CommandHandler.php')) {
        throw new Exception('CommandHandler.php not found');
    }
    
    if (!file_exists('includes/error_handler.php')) {
        throw new Exception('error_handler.php not found');
    }
    
    require_once 'handlers/CommandHandler.php';
    require_once 'includes/error_handler.php';
    
    // Ambil input dari Telegram
    $input = file_get_contents('php://input');
    
    // Handle direct browser access
    if (empty($input)) {
        // If accessed via browser (GET request), show status page
        if ($_SERVER['REQUEST_METHOD'] === 'GET') {
            showWebhookStatus();
            exit;
        }
        throw new Exception('Empty input received');
    }
    
    $update = json_decode($input, true);
    
    // Check JSON decode error
    if (json_last_error() !== JSON_ERROR_NONE) {
        throw new Exception('JSON decode error: ' . json_last_error_msg());
    }
    
    // Log update untuk debugging
    if ($update) {
        logWebhook($update);
    } else {
        logError("Failed to decode JSON: " . $input);
    }
    
    // Validasi update
    if (!$update || (!isset($update['message']) && !isset($update['callback_query']))) {
        http_response_code(400);
        logError("Invalid update received: " . $input);
        
        $response = ['error' => 'Invalid update', 'received' => $input];
        echo json_encode($response);
        exit;
    }
    
    // Handle command
    $commandHandler = new CommandHandler();
    $result = $commandHandler->handleCommand($update);
    
    // Log successful processing
    $logMessage = "Command processed successfully";
    if (isset($update['message']['text'])) {
        $logMessage .= " - Command: " . $update['message']['text'];
    }
    if (isset($update['message']['from']['username'])) {
        $logMessage .= " - User: @" . $update['message']['from']['username'];
    }
    logBotActivity($logMessage);
    
    // Response sukses
    http_response_code(200);
    $response = ['status' => 'ok', 'result' => $result];
    
    echo json_encode($response);
    
} catch (Exception $e) {
    // Log error
    $errorMessage = "Webhook error: " . $e->getMessage() . " - File: " . $e->getFile() . " - Line: " . $e->getLine();
    
    // Try to log error, but don't fail if logging fails
    try {
        logError($errorMessage . " - Trace: " . $e->getTraceAsString());
    } catch (Exception $logError) {
        // If logging fails, write to PHP error log
        error_log($errorMessage);
    }
    
    // Response error
    http_response_code(500);
    $response = ['error' => 'Internal server error', 'message' => $e->getMessage()];
    
    echo json_encode($response);
} catch (Error $e) {
    // Handle fatal errors
    $errorMessage = "Fatal error: " . $e->getMessage() . " - File: " . $e->getFile() . " - Line: " . $e->getLine();
    error_log($errorMessage);
    
    http_response_code(500);
    echo json_encode(['error' => 'Fatal error', 'message' => $e->getMessage()]);
}

function showWebhookStatus() {
    header('Content-Type: text/html; charset=utf-8');
    
    echo "<!DOCTYPE html>
<html>
<head>
    <title>Telegram Bot Webhook Status</title>
    <meta charset='utf-8'>
    <style>
        body { font-family: Arial, sans-serif; margin: 40px; line-height: 1.6; }
        .status { padding: 15px; border-radius: 5px; margin: 10px 0; }
        .success { background: #d4edda; color: #155724; border: 1px solid #c3e6cb; }
        .info { background: #d1ecf1; color: #0c5460; border: 1px solid #bee5eb; }
        .warning { background: #fff3cd; color: #856404; border: 1px solid #ffeaa7; }
        code { background: #f8f9fa; padding: 2px 6px; border-radius: 3px; }
    </style>
</head>
<body>
    <h1>🤖 Telegram Bot Webhook</h1>
    
    <div class='status success'>
        <h3>✅ Webhook Status: Active</h3>
        <p>Webhook endpoint is running and ready to receive Telegram updates.</p>
    </div>
    
    <div class='status info'>
        <h3>📋 Information</h3>
        <ul>
            <li><strong>Endpoint:</strong> <code>" . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'] . "</code></li>
            <li><strong>Method:</strong> POST (Telegram API calls)</li>
            <li><strong>Server Time:</strong> " . date('Y-m-d H:i:s T') . "</li>
            <li><strong>PHP Version:</strong> " . PHP_VERSION . "</li>
        </ul>
    </div>
    
    <div class='status warning'>
        <h3>⚠️ Note</h3>
        <p>This page is shown when accessing the webhook directly via browser. 
        Telegram will send POST requests to this endpoint with bot updates.</p>
    </div>
    
    <div class='status info'>
        <h3>🔧 Setup Instructions</h3>
        <ol>
            <li>Set this URL as your bot webhook: <code>https://" . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'] . "</code></li>
            <li>Use Telegram API: <code>https://api.telegram.org/bot[BOT_TOKEN]/setWebhook?url=[WEBHOOK_URL]</code></li>
            <li>Test your bot by sending <code>/start</code> command</li>
        </ol>
    </div>
    
    <hr>
    <p><small>Telegram Bot Webhook Endpoint - " . date('Y') . "</small></p>
</body>
</html>";
}
?>