<?php
// Enhanced webhook setter with troubleshooting for production
require_once 'config/telegram.php';

$webhookUrl = 'https://jajargenjang.web.id/webhook.php';

echo "<h1>🔗 Set Telegram Webhook</h1>";
echo "<hr>";

// Test 1: Check if webhook URL is accessible
echo "<h3>1. 🌐 Testing Webhook URL Accessibility</h3>";
$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, $webhookUrl);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_TIMEOUT, 10);
curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
curl_setopt($ch, CURLOPT_USERAGENT, 'TelegramBot/1.0');

$response = curl_exec($ch);
$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
$error = curl_error($ch);
curl_close($ch);

if ($httpCode == 200) {
    echo "<div style='background:#d4edda; padding:15px; border-radius:5px; margin:15px 0;'>";
    echo "<h4>✅ URL Accessible</h4>";
    echo "<p><strong>HTTP Code:</strong> $httpCode</p>";
    echo "<p><strong>Response Length:</strong> " . strlen($response) . " bytes</p>";
    echo "</div>";
} else {
    echo "<div style='background:#f8d7da; padding:15px; border-radius:5px; margin:15px 0;'>";
    echo "<h4>❌ URL Not Accessible</h4>";
    echo "<p><strong>HTTP Code:</strong> $httpCode</p>";
    echo "<p><strong>Error:</strong> $error</p>";
    echo "</div>";
}

// Test 2: DNS Resolution
echo "<h3>2. 🔍 DNS Resolution Test</h3>";
$domain = 'jajargenjang.web.id';
$ip = gethostbyname($domain);

if ($ip !== $domain) {
    echo "<div style='background:#d4edda; padding:15px; border-radius:5px; margin:15px 0;'>";
    echo "<h4>✅ DNS Resolution OK</h4>";
    echo "<p><strong>Domain:</strong> $domain</p>";
    echo "<p><strong>IP Address:</strong> $ip</p>";
    echo "</div>";
} else {
    echo "<div style='background:#f8d7da; padding:15px; border-radius:5px; margin:15px 0;'>";
    echo "<h4>❌ DNS Resolution Failed</h4>";
    echo "<p><strong>Domain:</strong> $domain</p>";
    echo "<p>Cannot resolve to IP address</p>";
    echo "</div>";
}

// Test 3: Try to set webhook
echo "<h3>3. 🔧 Setting Webhook</h3>";

try {
    $bot = new TelegramBot();
    
    // First, delete existing webhook to clear cache
    echo "<p>🗑️ Clearing existing webhook...</p>";
    $deleteResult = $bot->deleteWebhook();
    
    if ($deleteResult['ok']) {
        echo "<p style='color:#28a745;'>✅ Existing webhook cleared</p>";
    } else {
        echo "<p style='color:#dc3545;'>⚠️ Could not clear webhook: " . ($deleteResult['description'] ?? 'Unknown error') . "</p>";
    }
    
    // Wait a moment
    sleep(2);
    
    // Set new webhook
    echo "<p>🔗 Setting new webhook...</p>";
    $result = $bot->setWebhook($webhookUrl);
    
    if ($result['ok']) {
        echo "<div style='background:#d4edda; padding:15px; border-radius:5px; margin:15px 0;'>";
        echo "<h4>✅ Webhook Set Successfully!</h4>";
        echo "<p><strong>URL:</strong> $webhookUrl</p>";
        echo "<p><strong>Status:</strong> " . ($result['description'] ?? 'Active') . "</p>";
        echo "</div>";
    } else {
        echo "<div style='background:#f8d7da; padding:15px; border-radius:5px; margin:15px 0;'>";
        echo "<h4>❌ Failed to Set Webhook</h4>";
        echo "<p><strong>Error:</strong> " . ($result['description'] ?? 'Unknown error') . "</p>";
        echo "<p><strong>Error Code:</strong> " . ($result['error_code'] ?? 'N/A') . "</p>";
        echo "</div>";
        
        // Show troubleshooting tips
        echo "<div style='background:#fff3cd; padding:15px; border-radius:5px; margin:15px 0;'>";
        echo "<h4>🔧 Troubleshooting Tips:</h4>";
        echo "<ul>";
        echo "<li><strong>DNS Propagation:</strong> Wait 24-48 hours for full DNS propagation</li>";
        echo "<li><strong>SSL Certificate:</strong> Ensure your hosting has valid SSL certificate</li>";
        echo "<li><strong>Firewall:</strong> Check if hosting blocks Telegram IPs</li>";
        echo "<li><strong>Try Alternative:</strong> Use different subdomain or domain</li>";
        echo "</ul>";
        echo "</div>";
    }
    
} catch (Exception $e) {
    echo "<div style='background:#f8d7da; padding:15px; border-radius:5px; margin:15px 0;'>";
    echo "<h4>❌ Exception Occurred</h4>";
    echo "<p>" . htmlspecialchars($e->getMessage()) . "</p>";
    echo "</div>";
}

// Test 4: Get current webhook info
echo "<h3>4. 📊 Current Webhook Info</h3>";
try {
    $webhookInfo = $bot->getWebhookInfo();
    
    if ($webhookInfo['ok']) {
        $info = $webhookInfo['result'];
        echo "<div style='background:#f8f9fa; padding:15px; border-radius:5px;'>";
        echo "<strong>URL:</strong> " . ($info['url'] ?: 'Not set') . "<br>";
        echo "<strong>Pending Updates:</strong> " . ($info['pending_update_count'] ?? 0) . "<br>";
        
        if (isset($info['last_error_message'])) {
            echo "<strong style='color:#dc3545;'>Last Error:</strong> " . $info['last_error_message'] . "<br>";
            echo "<strong>Error Date:</strong> " . date('Y-m-d H:i:s', $info['last_error_date']) . "<br>";
        } else {
            echo "<strong style='color:#28a745;'>Status:</strong> No errors<br>";
        }
        
        echo "<strong>Max Connections:</strong> " . ($info['max_connections'] ?? 40) . "<br>";
        
        if (isset($info['ip_address'])) {
            echo "<strong>IP Address:</strong> " . $info['ip_address'] . "<br>";
        }
        
        echo "</div>";
    }
} catch (Exception $e) {
    echo "<p style='color:#dc3545;'>Error getting webhook info: " . htmlspecialchars($e->getMessage()) . "</p>";
}

// Alternative methods
echo "<hr>";
echo "<h3>🔄 Alternative Methods</h3>";
echo "<div style='background:#d1ecf1; padding:15px; border-radius:5px; margin:15px 0;'>";
echo "<h4>Method 1: Direct API Call</h4>";
echo "<p>Try this URL in browser:</p>";
echo "<code>https://api.telegram.org/bot" . BOT_TOKEN . "/setWebhook?url=" . urlencode($webhookUrl) . "</code>";
echo "</div>";

echo "<div style='background:#d1ecf1; padding:15px; border-radius:5px; margin:15px 0;'>";
echo "<h4>Method 2: Use Different Domain</h4>";
echo "<p>If DNS issues persist, try:</p>";
echo "<ul>";
echo "<li>Use IP address instead of domain (not recommended for production)</li>";
echo "<li>Use different subdomain: <code>bot.jajargenjang.web.id</code></li>";
echo "<li>Use different domain temporarily</li>";
echo "</ul>";
echo "</div>";

echo "<div style='background:#d1ecf1; padding:15px; border-radius:5px; margin:15px 0;'>";
echo "<h4>Method 3: Wait and Retry</h4>";
echo "<p>DNS propagation can take time. Try again in:</p>";
echo "<ul>";
echo "<li>15 minutes</li>";
echo "<li>1 hour</li>";
echo "<li>24 hours</li>";
echo "</ul>";
echo "</div>";

echo "<hr>";
echo "<div style='text-align:center; margin:20px 0;'>";
echo "<a href='webhook.php' style='padding:10px 20px; background:#007bff; color:white; text-decoration:none; border-radius:5px; margin:5px;'>🔍 Test Webhook</a>";
echo "<a href='admin/' style='padding:10px 20px; background:#28a745; color:white; text-decoration:none; border-radius:5px; margin:5px;'>⚙️ Admin Panel</a>";
echo "<button onclick='location.reload()' style='padding:10px 20px; background:#6c757d; color:white; border:none; border-radius:5px; margin:5px; cursor:pointer;'>🔄 Retry</button>";
echo "</div>";

echo "<style>
body { font-family: Arial, sans-serif; margin: 20px; line-height: 1.6; }
code { background: #f8f9fa; padding: 2px 6px; border-radius: 3px; font-family: monospace; word-break: break-all; }
</style>";
?>