<?php
require_once 'models/Verification.php';
require_once 'models/User.php';

class VerificationService {
    private $verification;
    private $user;
    private $apiBaseUrl;
    
    public function __construct($apiBaseUrl = null) {
        $this->verification = new Verification();
        $this->user = new User();
        
        // Get API base URL from GlobalSettings if not provided
        if (!$apiBaseUrl) {
            require_once __DIR__ . '/../models/GlobalSettings.php';
            $settings = new GlobalSettings();
            $apiBaseUrl = $settings->getApiBaseUrl();
        }
        
        $this->apiBaseUrl = rtrim($apiBaseUrl, '/');
    }
    
    public function submitBatchVerification($userId, $verificationIds, $hcaptchaToken, $turnstileToken = null, $programId = null, $useLucky = true) {
        $batchId = 'batch_' . uniqid();
        
        // Simpan batch verification ke database
        $this->verification->createBatchVerification($userId, $batchId, count($verificationIds));
        
        // Siapkan data sesuai API docs
        $data = [
            'hCaptchaToken' => $hcaptchaToken,
            'verificationIds' => $verificationIds,
            'useLucky' => $useLucky
        ];
        
        if ($turnstileToken) {
            $data['turnstileToken'] = $turnstileToken;
        }
        
        if ($programId) {
            $data['programId'] = $programId;
        }
        
        // Kirim request ke API
        $response = $this->makeApiRequest('/api/batch', $data, $hcaptchaToken);
        
        if ($response) {
            $this->verification->updateBatchStatus($batchId, 'processing');
            return [
                'success' => true,
                'batchId' => $batchId,
                'message' => 'Batch verification started successfully'
            ];
        }
        
        $this->verification->updateBatchStatus($batchId, 'error');
        return [
            'success' => false,
            'message' => 'Failed to start batch verification'
        ];
    }
    
    public function checkVerificationStatus($checkToken) {
        $data = ['checkToken' => $checkToken];
        
        $response = $this->makeApiRequest('/api/check-status', $data, null, false);
        
        if ($response && isset($response['currentStep'])) {
            return [
                'success' => true,
                'status' => $response['currentStep'],
                'message' => $response['message'] ?? ''
            ];
        }
        
        return [
            'success' => false,
            'message' => 'Failed to check status'
        ];
    }
    
    public function cancelVerification($verificationId, $csrfToken) {
        $data = ['verificationId' => $verificationId];
        
        $response = $this->makeApiRequest('/api/cancel', $data, $csrfToken);
        
        if ($response && isset($response['currentStep'])) {
            // Update status di database
            $this->verification->updateStatus($verificationId, $response['currentStep'], $response['message'] ?? '');
            
            return [
                'success' => true,
                'status' => $response['currentStep'],
                'message' => $response['message'] ?? 'Verification cancelled'
            ];
        }
        
        return [
            'success' => false,
            'message' => 'Failed to cancel verification'
        ];
    }
    
    public function getSystemStatus() {
        $response = $this->makeApiRequest('/api/status', [], null, false, 'GET');
        
        if ($response) {
            return [
                'success' => true,
                'data' => $response
            ];
        }
        
        return [
            'success' => false,
            'message' => 'Failed to get system status'
        ];
    }
    
    private function makeApiRequest($endpoint, $data = [], $csrfToken = null, $requiresCsrf = true, $method = 'POST') {
        $url = $this->apiBaseUrl . $endpoint;
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 60); // Increase timeout for SSE
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        
        $headers = ['Content-Type: application/json'];
        
        // For batch API, expect SSE response
        if ($endpoint === '/api/batch') {
            $headers[] = 'Accept: text/event-stream';
        }
        
        if ($requiresCsrf && $csrfToken) {
            $headers[] = 'X-CSRF-Token: ' . $csrfToken;
        }
        
        if ($method === 'POST') {
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        }
        
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        
        $result = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $contentType = curl_getinfo($ch, CURLINFO_CONTENT_TYPE);
        curl_close($ch);
        
        // Log for debugging
        error_log("API Request to $url - HTTP Code: $httpCode, Content-Type: $contentType");
        error_log("Request Data: " . json_encode($data));
        error_log("Response: " . substr($result, 0, 500));
        
        if ($httpCode === 200 && $result) {
            // Handle SSE response for batch API
            if ($endpoint === '/api/batch' && strpos($contentType, 'text/event-stream') !== false) {
                return $this->parseSSEResponse($result);
            }
            
            // Handle JSON response
            $decoded = json_decode($result, true);
            if ($decoded !== null) {
                return $decoded;
            }
        }
        
        return false;
    }
    
    private function parseSSEResponse($sseData) {
        $events = [];
        $lines = explode("\n", $sseData);
        $currentEvent = [];
        
        foreach ($lines as $line) {
            $line = trim($line);
            
            if (empty($line)) {
                if (!empty($currentEvent)) {
                    $events[] = $currentEvent;
                    $currentEvent = [];
                }
                continue;
            }
            
            if (strpos($line, 'event:') === 0) {
                $currentEvent['event'] = trim(substr($line, 6));
            } elseif (strpos($line, 'data:') === 0) {
                $data = trim(substr($line, 5));
                $decoded = json_decode($data, true);
                if ($decoded !== null) {
                    $currentEvent['data'] = $decoded;
                } else {
                    $currentEvent['data'] = $data;
                }
            }
        }
        
        // Add last event if exists
        if (!empty($currentEvent)) {
            $events[] = $currentEvent;
        }
        
        return $events;
    }
}
?>