<?php
require_once __DIR__ . '/../config/database.php';

class GlobalSettings {
    private $db;
    
    public function __construct() {
        $database = new Database();
        $this->db = $database->getConnection();
    }
    
    // Debug method to get connection
    public function getConnection() {
        return $this->db;
    }
    
    public function getSetting($key, $default = null) {
        try {
            $stmt = $this->db->prepare("SELECT setting_value FROM global_settings WHERE setting_key = ?");
            $stmt->execute([$key]);
            $result = $stmt->fetch();
            
            return $result ? $result['setting_value'] : $default;
        } catch (Exception $e) {
            error_log("GlobalSettings::getSetting error: " . $e->getMessage() . " - Key: $key");
            return $default;
        }
    }
    
    public function setSetting($key, $value, $description = null) {
        try {
            $stmt = $this->db->prepare("
                INSERT INTO global_settings (setting_key, setting_value, description) 
                VALUES (?, ?, ?) 
                ON DUPLICATE KEY UPDATE 
                setting_value = VALUES(setting_value),
                description = COALESCE(VALUES(description), description),
                updated_at = NOW()
            ");
            
            return $stmt->execute([$key, $value, $description]);
        } catch (Exception $e) {
            error_log("GlobalSettings::setSetting error: " . $e->getMessage() . " - Key: $key");
            return false;
        }
    }
    
    public function getAllSettings() {
        try {
            $stmt = $this->db->query("SELECT * FROM global_settings ORDER BY setting_key");
            return $stmt->fetchAll();
        } catch (Exception $e) {
            error_log("GlobalSettings::getAllSettings error: " . $e->getMessage());
            return [];
        }
    }
    
    public function getTokens() {
        return [
            'hcaptcha_token' => $this->getSetting('hcaptcha_token'),
            'turnstile_token' => $this->getSetting('turnstile_token')
        ];
    }
    
    public function setTokens($hcaptchaToken, $turnstileToken = null) {
        $success = true;
        
        if ($hcaptchaToken !== null) {
            $success &= $this->setSetting('hcaptcha_token', $hcaptchaToken, 'Global HCaptcha token for all users');
        }
        
        if ($turnstileToken !== null) {
            $success &= $this->setSetting('turnstile_token', $turnstileToken, 'Global Turnstile token for all users');
        }
        
        return $success;
    }
    
    public function getBotStatus() {
        return $this->getSetting('bot_status', 'active');
    }
    
    public function setBotStatus($status) {
        return $this->setSetting('bot_status', $status, 'Bot status: active, maintenance');
    }
    
    public function getApiBaseUrl() {
        return $this->getSetting('api_base_url', 'https://neigui.1key.me');
    }
    
    public function setApiBaseUrl($url) {
        return $this->setSetting('api_base_url', $url, 'Base URL for verification API');
    }
    
    public function getMaxVerificationsPerUser() {
        return (int)$this->getSetting('max_verifications_per_user', 10);
    }
    
    public function setMaxVerificationsPerUser($max) {
        return $this->setSetting('max_verifications_per_user', $max, 'Maximum verifications per user per day');
    }
    
    public function deleteSetting($key) {
        try {
            $stmt = $this->db->prepare("DELETE FROM global_settings WHERE setting_key = ?");
            return $stmt->execute([$key]);
        } catch (Exception $e) {
            error_log("GlobalSettings::deleteSetting error: " . $e->getMessage());
            return false;
        }
    }
}
?>