<?php
require_once 'config/telegram.php';
require_once 'config/admin_bot.php';
require_once 'models/User.php';
require_once 'services/VerificationService.php';

class CommandHandler {
    private $bot;
    private $user;
    private $verificationService;
    
    public function __construct() {
        $this->bot = new TelegramBot();
        $this->user = new User();
        $this->verificationService = new VerificationService();
    }
    
    private function formatMessage($message) {
        // Convert literal \n to actual line breaks
        return str_replace('\\n', "\n", $message);
    }
    
    public function handleCommand($update) {
        $message = $update['message'] ?? $update['callback_query']['message'] ?? null;
        $chatId = $message['chat']['id'];
        $userId = $message['from']['id'];
        $text = $message['text'] ?? '';
        
        // Simpan/update user data
        $userData = [
            'username' => $message['from']['username'] ?? null,
            'first_name' => $message['from']['first_name'] ?? null,
            'last_name' => $message['from']['last_name'] ?? null
        ];
        $user = $this->user->findOrCreate($userId, $userData);
        
        // Handle callback query
        if (isset($update['callback_query'])) {
            return $this->handleCallbackQuery($update['callback_query'], $user);
        }
        
        // Handle text commands
        if (strpos($text, '/start') === 0) {
            return $this->handleStart($chatId, $userId);
        } elseif (strpos($text, '/help') === 0) {
            return $this->handleHelp($chatId);
        } elseif (strpos($text, '/verify') === 0) {
            return $this->handleVerify($chatId, $text, $user);
        } elseif (strpos($text, '/batch') === 0) {
            return $this->handleBatch($chatId, $text, $user);
        } elseif (strpos($text, '/status') === 0) {
            return $this->handleStatus($chatId, $text, $user);
        } elseif (strpos($text, '/myverifications') === 0) {
            return $this->handleMyVerifications($chatId, $user);
        } elseif (strpos($text, '/admin') === 0) {
            return $this->handleAdminCommands($chatId, $text, $user, $userId);
        } else {
            return $this->handleUnknownCommand($chatId);
        }
    }
    
    private function handleStart($chatId, $userId) {
        $adminLevel = BotAdminAuth::getAdminLevel($userId);
        $isAdmin = BotAdminAuth::isAdmin($userId);
        
        // Cek status bot
        require_once __DIR__ . '/../models/GlobalSettings.php';
        $settings = new GlobalSettings();
        $botStatus = $settings->getBotStatus();
        
        if ($botStatus === 'maintenance' && !$isAdmin) {
            $maintenanceMessage = $settings->getSetting('maintenance_message', '🔧 <b>Bot sedang dalam maintenance</b>\n\nSilakan coba lagi nanti.\nTerima kasih atas pengertiannya.');
            $maintenanceMessage = str_replace('\\n', "\n", $maintenanceMessage);
            return $this->bot->sendMessage($chatId, $maintenanceMessage);
        }
        
        // Get custom start message
        $startMessage = $settings->getSetting('start_message', '🤖 <b>Selamat datang di Verification Bot!</b>\n\nBot ini membantu Anda melakukan verifikasi menggunakan API.');
        $startMessage = $this->formatMessage($startMessage);
        
        $text = $startMessage . "\n\n";
        
        if ($isAdmin) {
            $text .= "👑 <b>Status: " . ($adminLevel === 'super_admin' ? 'Super Admin' : 'Admin') . "</b>\n\n";
        }
        
        // Cek apakah token sudah diset
        $tokens = $settings->getTokens();
        if (!$tokens['hcaptcha_token']) {
            if ($isAdmin) {
                $text .= "⚠️ <i>Token API belum diset. Silakan set melalui panel admin.</i>\n\n";
            } else {
                $text .= "⚠️ <i>Bot belum dikonfigurasi. Hubungi admin.</i>\n\n";
            }
        } else {
            $text .= "✅ <i>Bot siap digunakan untuk verifikasi!</i>\n\n";
        }
        
        $text .= "Pilih menu di bawah ini:";
        
        // Build main menu keyboard
        $keyboard = ['inline_keyboard' => []];
        
        // First row: Help & Status
        $firstRow = [];
        if ($settings->getSetting('show_help_button', '1')) {
            $firstRow[] = ['text' => '❓ Help & Panduan', 'callback_data' => 'help'];
        }
        if ($settings->getSetting('show_status_button', '1')) {
            $firstRow[] = ['text' => '📊 Status Sistem', 'callback_data' => 'status'];
        }
        if (!empty($firstRow)) {
            $keyboard['inline_keyboard'][] = $firstRow;
        }
        
        // Second row: Verification buttons
        $secondRow = [];
        if ($settings->getSetting('show_verify_button', '1')) {
            $secondRow[] = ['text' => '✅ Verifikasi Single', 'callback_data' => 'verify_menu'];
        }
        if ($settings->getSetting('show_batch_button', '1')) {
            $secondRow[] = ['text' => '📦 Verifikasi Batch', 'callback_data' => 'batch_menu'];
        }
        if (!empty($secondRow)) {
            $keyboard['inline_keyboard'][] = $secondRow;
        }
        
        // Third row: History
        if ($settings->getSetting('show_history_button', '1')) {
            $keyboard['inline_keyboard'][] = [
                ['text' => '📋 Riwayat Verifikasi', 'callback_data' => 'myverifications']
            ];
        }
        
        // Admin buttons
        if ($isAdmin) {
            $showAdminButton = $settings->getSetting('show_admin_button', '1');
            $showWebAdminLink = $settings->getSetting('show_web_admin_link', '1');
            
            if ($showAdminButton || $showWebAdminLink) {
                $adminRow = [];
                if ($showAdminButton) {
                    $adminRow[] = ['text' => '👑 Admin Panel', 'callback_data' => 'admin_panel'];
                }
                if ($showWebAdminLink) {
                    $adminRow[] = ['text' => '🌐 Web Admin', 'url' => 'https://jajargenjang.web.id/admin/'];
                }
                $keyboard['inline_keyboard'][] = $adminRow;
            }
        }
        
        // Add custom buttons if configured
        $customButtons = $settings->getSetting('custom_menu_buttons', '');
        if ($customButtons) {
            try {
                $customButtonsArray = json_decode($customButtons, true);
                if (is_array($customButtonsArray)) {
                    foreach ($customButtonsArray as $button) {
                        if (isset($button['text']) && (isset($button['callback_data']) || isset($button['url']))) {
                            $keyboard['inline_keyboard'][] = [$button];
                        }
                    }
                }
            } catch (Exception $e) {
                // Ignore invalid JSON
            }
        }
        
        return $this->bot->sendMessage($chatId, $text, $keyboard);
    }
    
    private function handleHelp($chatId) {
        require_once __DIR__ . '/../models/GlobalSettings.php';
        $settings = new GlobalSettings();
        
        $helpMessage = $settings->getSetting('help_message', '📖 <b>Bantuan Verification Bot</b>\n\nGunakan menu button atau command berikut untuk berinteraksi dengan bot:');
        $helpMessage = $this->formatMessage($helpMessage);
        
        $text = $helpMessage . "\n\n";
        
        $text .= "<b>🎯 Cara Menggunakan Bot:</b>\n";
        $text .= "1. Gunakan <b>button menu</b> di pesan /start\n";
        $text .= "2. Atau ketik command langsung\n\n";
        
        $text .= "<b>📋 Daftar Command:</b>\n";
        $text .= "/start - Menu utama (dengan button)\n";
        $text .= "/help - Bantuan dan panduan\n";
        $text .= "/verify ID - Verifikasi single ID\n";
        $text .= "/batch ID1,ID2,ID3 - Verifikasi multiple IDs\n";
        $text .= "/status - Cek status sistem\n";
        $text .= "/myverifications - Lihat riwayat verifikasi\n\n";
        
        $text .= "<b>✅ Contoh Verifikasi:</b>\n";
        $text .= "<code>/verify vid_123456</code>\n";
        $text .= "<code>/batch vid_123,vid_456,vid_789</code>\n\n";
        
        $text .= "<b>💡 Tips:</b>\n";
        $text .= "• Gunakan button untuk navigasi yang mudah\n";
        $text .= "• Copy-paste verification ID untuk akurasi\n";
        $text .= "• Cek /myverifications untuk melihat riwayat\n\n";
        
        $text .= "<b>🔧 Butuh Bantuan?</b>\n";
        $text .= "Hubungi admin jika mengalami masalah.";
        
        $keyboard = [
            'inline_keyboard' => [
                [
                    ['text' => '🔙 Kembali ke Menu', 'callback_data' => 'back_to_main']
                ]
            ]
        ];
        
        return $this->bot->sendMessage($chatId, $text, $keyboard);
    }
    
    private function handleVerify($chatId, $text, $user) {
        // Cek status bot
        require_once __DIR__ . '/../models/GlobalSettings.php';
        $settings = new GlobalSettings();
        
        if ($settings->getBotStatus() === 'maintenance') {
            $maintenanceMessage = $settings->getSetting('maintenance_message', '🔧 <b>Bot sedang dalam maintenance</b>\n\nSilakan coba lagi nanti.');
            return $this->bot->sendMessage($chatId, $maintenanceMessage);
        }
        
        $parts = explode(' ', trim($text), 2);
        
        if (count($parts) < 2) {
            $text = "❌ <b>Format salah!</b>\n\n";
            $text .= "Gunakan: <code>/verify VERIFICATION_ID</code>\n";
            $text .= "Contoh: <code>/verify vid_123456</code>";
            return $this->bot->sendMessage($chatId, $text);
        }
        
        $verificationId = $parts[1];
        
        // Ambil token global
        $tokens = $settings->getTokens();
        
        if (!$tokens['hcaptcha_token']) {
            $text = "❌ <b>Bot belum dikonfigurasi!</b>\n\n";
            $text .= "Token API belum diset oleh administrator.\n";
            $text .= "Silakan hubungi admin untuk mengaktifkan bot.";
            return $this->bot->sendMessage($chatId, $text);
        }
        
        // Untuk single verification, kita gunakan batch dengan 1 item
        $result = $this->verificationService->submitBatchVerification(
            $user['id'],
            [$verificationId],
            $tokens['hcaptcha_token'],
            $tokens['turnstile_token']
        );
        
        if ($result['success']) {
            $text = "✅ <b>Verifikasi dimulai!</b>\n\n";
            $text .= "ID: <code>$verificationId</code>\n";
            $text .= "Batch ID: <code>{$result['batchId']}</code>\n\n";
            $text .= "Proses sedang berjalan... Anda akan mendapat notifikasi saat selesai.";
        } else {
            $text = "❌ <b>Gagal memulai verifikasi!</b>\n\n";
            $text .= $result['message'];
        }
        
        return $this->bot->sendMessage($chatId, $text);
    }
    
    private function handleBatch($chatId, $text, $user) {
        // Cek status bot
        require_once __DIR__ . '/../models/GlobalSettings.php';
        $settings = new GlobalSettings();
        
        if ($settings->getBotStatus() === 'maintenance') {
            $maintenanceMessage = $settings->getSetting('maintenance_message', '🔧 <b>Bot sedang dalam maintenance</b>\n\nSilakan coba lagi nanti.');
            return $this->bot->sendMessage($chatId, $maintenanceMessage);
        }
        
        $parts = explode(' ', trim($text), 2);
        
        if (count($parts) < 2) {
            $text = "❌ <b>Format salah!</b>\n\n";
            $text .= "Gunakan: <code>/batch ID1,ID2,ID3</code>\n";
            $text .= "Contoh: <code>/batch vid_123,vid_456,vid_789</code>";
            return $this->bot->sendMessage($chatId, $text);
        }
        
        $verificationIds = array_map('trim', explode(',', $parts[1]));
        
        // Ambil token global
        $tokens = $settings->getTokens();
        
        if (!$tokens['hcaptcha_token']) {
            $text = "❌ <b>Bot belum dikonfigurasi!</b>\n\n";
            $text .= "Token API belum diset oleh administrator.\n";
            $text .= "Silakan hubungi admin untuk mengaktifkan bot.";
            return $this->bot->sendMessage($chatId, $text);
        }
        
        $maxVerifications = $settings->getMaxVerificationsPerUser();
        if (count($verificationIds) > $maxVerifications) {
            $text = "❌ <b>Terlalu banyak ID!</b>\n\n";
            $text .= "Maksimal $maxVerifications ID per batch.";
            return $this->bot->sendMessage($chatId, $text);
        }
        
        $result = $this->verificationService->submitBatchVerification(
            $user['id'],
            $verificationIds,
            $tokens['hcaptcha_token'],
            $tokens['turnstile_token']
        );
        
        if ($result['success']) {
            $text = "✅ <b>Batch verifikasi dimulai!</b>\n\n";
            $text .= "Total ID: " . count($verificationIds) . "\n";
            $text .= "Batch ID: <code>{$result['batchId']}</code>\n\n";
            $text .= "Proses sedang berjalan... Anda akan mendapat notifikasi saat selesai.";
        } else {
            $text = "❌ <b>Gagal memulai batch verifikasi!</b>\n\n";
            $text .= $result['message'];
        }
        
        return $this->bot->sendMessage($chatId, $text);
    }
    
    private function handleStatus($chatId, $text, $user) {
        $parts = explode(' ', trim($text), 2);
        
        if (count($parts) < 2) {
            // Tampilkan status sistem
            $result = $this->verificationService->getSystemStatus();
            
            if ($result['success']) {
                $text = "🔍 <b>Status Sistem</b>\n\n";
                $text .= "<pre>" . json_encode($result['data'], JSON_PRETTY_PRINT) . "</pre>";
            } else {
                $text = "❌ <b>Gagal mendapatkan status sistem!</b>";
            }
            
            return $this->bot->sendMessage($chatId, $text);
        }
        
        $checkToken = $parts[1];
        $result = $this->verificationService->checkVerificationStatus($checkToken);
        
        if ($result['success']) {
            $text = "🔍 <b>Status Verifikasi</b>\n\n";
            $text .= "Status: <b>{$result['status']}</b>\n";
            if ($result['message']) {
                $text .= "Pesan: {$result['message']}\n";
            }
        } else {
            $text = "❌ <b>Gagal cek status!</b>\n\n";
            $text .= $result['message'];
        }
        
        return $this->bot->sendMessage($chatId, $text);
    }
    
    private function handleMyVerifications($chatId, $user) {
        $verification = new Verification();
        $verifications = $verification->getUserVerifications($user['id'], 5);
        
        if (empty($verifications)) {
            $text = "📋 <b>Verifikasi Saya</b>\n\n";
            $text .= "Belum ada verifikasi yang dilakukan.";
            return $this->bot->sendMessage($chatId, $text);
        }
        
        $text = "📋 <b>Verifikasi Saya (5 Terakhir)</b>\n\n";
        
        foreach ($verifications as $v) {
            $statusIcon = $this->getStatusIcon($v['status']);
            $text .= "$statusIcon <code>{$v['verification_id']}</code>\n";
            $text .= "Status: <b>{$v['status']}</b>\n";
            if ($v['message']) {
                $text .= "Pesan: {$v['message']}\n";
            }
            $text .= "Waktu: " . date('d/m/Y H:i', strtotime($v['created_at'])) . "\n\n";
        }
        
        return $this->bot->sendMessage($chatId, $text);
    }
    
    private function handleCallbackQuery($callbackQuery, $user) {
        $chatId = $callbackQuery['message']['chat']['id'];
        $messageId = $callbackQuery['message']['message_id'];
        $data = $callbackQuery['data'];
        $userId = $callbackQuery['from']['id'];
        
        switch ($data) {
            case 'help':
                return $this->handleHelp($chatId);
                
            case 'status':
                return $this->handleStatus($chatId, '/status', $user);
                
            case 'verify_menu':
                return $this->showVerifyMenu($chatId);
                
            case 'batch_menu':
                return $this->showBatchMenu($chatId);
                
            case 'myverifications':
                return $this->handleMyVerifications($chatId, $user);
                
            case 'back_to_main':
                return $this->handleBackToMain($chatId, $userId);
                
            case 'admin_panel':
                if (!BotAdminAuth::isAdmin($userId)) {
                    $text = "❌ <b>Akses Ditolak!</b>\n\nHanya admin yang dapat mengakses panel admin.";
                } else {
                    return $this->handleAdminMenu($chatId, $userId);
                }
                break;
                
            case 'admin_stats':
                if (!BotAdminAuth::isAdmin($userId)) {
                    $text = "❌ Akses ditolak.";
                } else {
                    return $this->handleAdminStats($chatId);
                }
                break;
                
            case 'admin_users':
                if (!BotAdminAuth::isAdmin($userId)) {
                    $text = "❌ Akses ditolak.";
                } else {
                    return $this->handleAdminUsers($chatId);
                }
                break;
                
            case 'admin_tokens':
                if (!BotAdminAuth::isAdmin($userId)) {
                    $text = "❌ Akses ditolak.";
                } else {
                    return $this->handleAdminTokens($chatId);
                }
                break;
                
            default:
                $text = "❓ Perintah tidak dikenal.";
        }
        
        return $this->bot->editMessage($chatId, $messageId, $text);
    }
    
    private function handleUnknownCommand($chatId) {
        require_once __DIR__ . '/../models/GlobalSettings.php';
        $settings = new GlobalSettings();
        
        $unknownMessage = $settings->getSetting('unknown_command_message', '❓ <b>Perintah tidak dikenal!</b>\n\nGunakan /help untuk melihat daftar perintah yang tersedia.');
        $unknownMessage = $this->formatMessage($unknownMessage);
        
        return $this->bot->sendMessage($chatId, $unknownMessage);
    }
    
    private function showVerifyMenu($chatId) {
        require_once __DIR__ . '/../models/GlobalSettings.php';
        $settings = new GlobalSettings();
        
        $text = "✅ <b>Menu Verifikasi Single</b>\n\n";
        $text .= "Untuk melakukan verifikasi single ID, silakan kirim pesan dengan format:\n\n";
        $text .= "<code>/verify VERIFICATION_ID</code>\n\n";
        $text .= "<b>Contoh:</b>\n";
        $text .= "<code>/verify vid_123456</code>\n";
        $text .= "<code>/verify abc123def456</code>\n\n";
        $text .= "💡 <i>Tip: Copy verification ID dari email atau link yang Anda terima</i>";
        
        $keyboard = [
            'inline_keyboard' => [
                [
                    ['text' => '🔙 Kembali ke Menu', 'callback_data' => 'back_to_main']
                ]
            ]
        ];
        
        return $this->bot->sendMessage($chatId, $text, $keyboard);
    }
    
    private function showBatchMenu($chatId) {
        require_once __DIR__ . '/../models/GlobalSettings.php';
        $settings = new GlobalSettings();
        
        $maxVerifications = $settings->getMaxVerificationsPerUser();
        
        $text = "📦 <b>Menu Verifikasi Batch</b>\n\n";
        $text .= "Untuk melakukan verifikasi multiple IDs sekaligus, silakan kirim pesan dengan format:\n\n";
        $text .= "<code>/batch ID1,ID2,ID3</code>\n\n";
        $text .= "<b>Contoh:</b>\n";
        $text .= "<code>/batch vid_123,vid_456,vid_789</code>\n\n";
        $text .= "<b>Ketentuan:</b>\n";
        $text .= "• Maksimal <b>$maxVerifications</b> ID per batch\n";
        $text .= "• Pisahkan ID dengan koma (,)\n";
        $text .= "• Tidak boleh ada spasi setelah koma\n\n";
        $text .= "💡 <i>Tip: Pastikan semua ID valid sebelum mengirim</i>";
        
        $keyboard = [
            'inline_keyboard' => [
                [
                    ['text' => '🔙 Kembali ke Menu', 'callback_data' => 'back_to_main']
                ]
            ]
        ];
        
        return $this->bot->sendMessage($chatId, $text, $keyboard);
    }
    
    private function handleBackToMain($chatId, $userId) {
        // Redirect to start command
        return $this->handleStart($chatId, $userId);
    }
    
    private function handleAdminCommands($chatId, $text, $user, $userId) {
        if (!BotAdminAuth::isAdmin($userId)) {
            $text = "❌ <b>Akses Ditolak!</b>\n\n";
            $text .= "Perintah admin hanya tersedia untuk administrator.\n";
            $text .= "Your ID: <code>$userId</code>";
            return $this->bot->sendMessage($chatId, $text);
        }
        
        $parts = explode(' ', trim($text), 2);
        $subCommand = $parts[1] ?? '';
        
        switch ($subCommand) {
            case 'stats':
                return $this->handleAdminStats($chatId);
            case 'users':
                return $this->handleAdminUsers($chatId);
            case 'tokens':
                return $this->handleAdminTokens($chatId);
            default:
                return $this->handleAdminMenu($chatId, $userId);
        }
    }
    
    private function handleAdminMenu($chatId, $userId) {
        $isSuperAdmin = BotAdminAuth::isSuperAdmin($userId);
        
        $text = "👑 <b>Admin Panel</b>\n\n";
        $text .= "Pilih menu admin:\n\n";
        $text .= "/admin stats - Statistik bot\n";
        $text .= "/admin users - Kelola users\n";
        $text .= "/admin tokens - Kelola tokens\n";
        
        if ($isSuperAdmin) {
            $text .= "\n<b>Super Admin Only:</b>\n";
            $text .= "• Akses penuh ke panel web admin\n";
            $text .= "• Kelola admin lainnya\n";
        }
        
        $keyboard = [
            'inline_keyboard' => [
                [
                    ['text' => '📊 Stats', 'callback_data' => 'admin_stats'],
                    ['text' => '👥 Users', 'callback_data' => 'admin_users']
                ],
                [
                    ['text' => '🔑 Tokens', 'callback_data' => 'admin_tokens'],
                    ['text' => '🌐 Web Panel', 'url' => 'https://yourdomain.com/bot/admin/']
                ]
            ]
        ];
        
        return $this->bot->sendMessage($chatId, $text, $keyboard);
    }
    
    private function handleAdminStats($chatId) {
        try {
            require_once 'config/database.php';
            $db = new Database();
            $conn = $db->getConnection();
            
            // Get statistics
            $stats = [];
            
            $stmt = $conn->query("SELECT COUNT(*) as total FROM users");
            $stats['total_users'] = $stmt->fetch()['total'];
            
            $stmt = $conn->query("SELECT COUNT(*) as total FROM verifications");
            $stats['total_verifications'] = $stmt->fetch()['total'];
            
            $stmt = $conn->query("SELECT COUNT(*) as total FROM verifications WHERE status = 'success'");
            $stats['success_verifications'] = $stmt->fetch()['total'];
            
            $stmt = $conn->query("SELECT COUNT(*) as total FROM verifications WHERE DATE(created_at) = CURDATE()");
            $stats['today_verifications'] = $stmt->fetch()['total'];
            
            $text = "📊 <b>Bot Statistics</b>\n\n";
            $text .= "👥 Total Users: <b>" . number_format($stats['total_users']) . "</b>\n";
            $text .= "✅ Total Verifications: <b>" . number_format($stats['total_verifications']) . "</b>\n";
            $text .= "🎯 Success Rate: <b>" . ($stats['total_verifications'] > 0 ? round(($stats['success_verifications'] / $stats['total_verifications']) * 100, 1) : 0) . "%</b>\n";
            $text .= "📅 Today: <b>" . number_format($stats['today_verifications']) . "</b> verifications\n";
            
            return $this->bot->sendMessage($chatId, $text);
            
        } catch (Exception $e) {
            $text = "❌ <b>Error getting statistics:</b>\n" . $e->getMessage();
            return $this->bot->sendMessage($chatId, $text);
        }
    }
    
    private function handleAdminUsers($chatId) {
        try {
            require_once 'config/database.php';
            $db = new Database();
            $conn = $db->getConnection();
            
            $stmt = $conn->query("
                SELECT u.*, COUNT(v.id) as verification_count 
                FROM users u 
                LEFT JOIN verifications v ON u.id = v.user_id 
                GROUP BY u.id 
                ORDER BY u.created_at DESC 
                LIMIT 10
            ");
            $users = $stmt->fetchAll();
            
            $text = "👥 <b>Recent Users (Last 10)</b>\n\n";
            
            foreach ($users as $user) {
                $text .= "👤 <b>" . htmlspecialchars($user['first_name'] ?? 'Unknown') . "</b>\n";
                $text .= "   ID: <code>" . $user['telegram_id'] . "</code>\n";
                $text .= "   Username: @" . ($user['username'] ?? 'N/A') . "\n";
                $text .= "   Verifications: " . $user['verification_count'] . "\n";
                $text .= "   Joined: " . date('d/m/Y', strtotime($user['created_at'])) . "\n\n";
            }
            
            return $this->bot->sendMessage($chatId, $text);
            
        } catch (Exception $e) {
            $text = "❌ <b>Error getting users:</b>\n" . $e->getMessage();
            return $this->bot->sendMessage($chatId, $text);
        }
    }
    
    private function handleAdminTokens($chatId) {
        try {
            require_once 'config/database.php';
            $db = new Database();
            $conn = $db->getConnection();
            
            $stmt = $conn->query("
                SELECT ut.*, u.first_name, u.username 
                FROM user_tokens ut 
                JOIN users u ON ut.user_id = u.id 
                ORDER BY ut.created_at DESC 
                LIMIT 5
            ");
            $tokens = $stmt->fetchAll();
            
            $text = "🔑 <b>Token Management</b>\n\n";
            
            if (empty($tokens)) {
                $text .= "⚠️ Belum ada token yang diset.\n";
                $text .= "Gunakan /settoken untuk mengatur token API.";
            } else {
                $text .= "📋 <b>Active Tokens:</b>\n\n";
                
                foreach ($tokens as $token) {
                    $text .= "👤 <b>" . htmlspecialchars($token['first_name']) . "</b>\n";
                    $text .= "   HCaptcha: <code>" . substr($token['hcaptcha_token'], 0, 10) . "...</code>\n";
                    if ($token['turnstile_token']) {
                        $text .= "   Turnstile: <code>" . substr($token['turnstile_token'], 0, 10) . "...</code>\n";
                    }
                    $text .= "   Set: " . date('d/m/Y H:i', strtotime($token['created_at'])) . "\n\n";
                }
            }
            
            return $this->bot->sendMessage($chatId, $text);
            
        } catch (Exception $e) {
            $text = "❌ <b>Error getting tokens:</b>\n" . $e->getMessage();
            return $this->bot->sendMessage($chatId, $text);
        }
    }
    
    private function getStatusIcon($status) {
        switch ($status) {
            case 'success': return '✅';
            case 'error': return '❌';
            case 'cancelled': return '🚫';
            case 'processing': return '⏳';
            case 'pending': return '⏸️';
            default: return '❓';
        }
    }
}
?>