<?php
require_once '../config/database.php';
require_once 'config/admin.php';

AdminAuth::requireLogin();

$pageTitle = 'Manajemen Verifikasi';
$db = new Database();
$conn = $db->getConnection();

// Handle actions
if ($_POST) {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'update_status':
                $verificationId = $_POST['verification_id'];
                $status = $_POST['status'];
                $message = $_POST['message'] ?? '';
                
                $stmt = $conn->prepare("UPDATE verifications SET status = ?, message = ?, updated_at = NOW() WHERE verification_id = ?");
                if ($stmt->execute([$status, $message, $verificationId])) {
                    $success = "Status verifikasi berhasil diperbarui!";
                } else {
                    $error = "Gagal memperbarui status verifikasi!";
                }
                break;
                
            case 'delete_verification':
                $verificationId = $_POST['verification_id'];
                $stmt = $conn->prepare("DELETE FROM verifications WHERE verification_id = ?");
                if ($stmt->execute([$verificationId])) {
                    $success = "Verifikasi berhasil dihapus!";
                } else {
                    $error = "Gagal menghapus verifikasi!";
                }
                break;
        }
    }
}

// Filters
$status = $_GET['status'] ?? '';
$search = $_GET['search'] ?? '';
$dateFrom = $_GET['date_from'] ?? '';
$dateTo = $_GET['date_to'] ?? '';

// Build WHERE clause
$whereConditions = [];
$params = [];

if ($status) {
    $whereConditions[] = "v.status = ?";
    $params[] = $status;
}

if ($search) {
    $whereConditions[] = "(v.verification_id LIKE ? OR u.first_name LIKE ? OR u.username LIKE ?)";
    $params[] = "%$search%";
    $params[] = "%$search%";
    $params[] = "%$search%";
}

if ($dateFrom) {
    $whereConditions[] = "DATE(v.created_at) >= ?";
    $params[] = $dateFrom;
}

if ($dateTo) {
    $whereConditions[] = "DATE(v.created_at) <= ?";
    $params[] = $dateTo;
}

$whereClause = !empty($whereConditions) ? 'WHERE ' . implode(' AND ', $whereConditions) : '';

// Pagination
$page = $_GET['page'] ?? 1;
$limit = ITEMS_PER_PAGE;
$offset = ($page - 1) * $limit;

// Get total count
$countQuery = "SELECT COUNT(*) as total FROM verifications v JOIN users u ON v.user_id = u.id $whereClause";
$stmt = $conn->prepare($countQuery);
$stmt->execute($params);
$totalVerifications = $stmt->fetch()['total'];
$totalPages = ceil($totalVerifications / $limit);

// Get verifications
$query = "
    SELECT v.*, u.first_name, u.last_name, u.username, u.telegram_id
    FROM verifications v 
    JOIN users u ON v.user_id = u.id 
    $whereClause
    ORDER BY v.created_at DESC 
    LIMIT $limit OFFSET $offset
";

$stmt = $conn->prepare($query);
$stmt->execute($params);
$verifications = $stmt->fetchAll();

// Get status counts for filter buttons
$statusCounts = [];
$statusQuery = "SELECT status, COUNT(*) as count FROM verifications GROUP BY status";
$stmt = $conn->query($statusQuery);
while ($row = $stmt->fetch()) {
    $statusCounts[$row['status']] = $row['count'];
}

include 'includes/header.php';
?>

<!-- Filters -->
<div class="card mb-4">
    <div class="card-body">
        <form method="GET" class="row g-3">
            <div class="col-md-3">
                <label class="form-label">Cari</label>
                <input type="text" class="form-control" name="search" 
                       value="<?php echo htmlspecialchars($search); ?>" 
                       placeholder="ID, nama, atau username">
            </div>
            
            <div class="col-md-2">
                <label class="form-label">Status</label>
                <select class="form-select" name="status">
                    <option value="">Semua Status</option>
                    <option value="pending" <?php echo $status === 'pending' ? 'selected' : ''; ?>>Pending</option>
                    <option value="processing" <?php echo $status === 'processing' ? 'selected' : ''; ?>>Processing</option>
                    <option value="success" <?php echo $status === 'success' ? 'selected' : ''; ?>>Success</option>
                    <option value="error" <?php echo $status === 'error' ? 'selected' : ''; ?>>Error</option>
                    <option value="cancelled" <?php echo $status === 'cancelled' ? 'selected' : ''; ?>>Cancelled</option>
                </select>
            </div>
            
            <div class="col-md-2">
                <label class="form-label">Dari Tanggal</label>
                <input type="date" class="form-control" name="date_from" 
                       value="<?php echo htmlspecialchars($dateFrom); ?>">
            </div>
            
            <div class="col-md-2">
                <label class="form-label">Sampai Tanggal</label>
                <input type="date" class="form-control" name="date_to" 
                       value="<?php echo htmlspecialchars($dateTo); ?>">
            </div>
            
            <div class="col-md-3">
                <label class="form-label">&nbsp;</label>
                <div class="d-flex gap-2">
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-search me-1"></i>Filter
                    </button>
                    <a href="verifications.php" class="btn btn-outline-secondary">
                        <i class="fas fa-times me-1"></i>Reset
                    </a>
                </div>
            </div>
        </form>
    </div>
</div>

<?php if (isset($success)): ?>
    <div class="alert alert-success alert-dismissible fade show">
        <i class="fas fa-check-circle me-2"></i><?php echo $success; ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
<?php endif; ?>

<?php if (isset($error)): ?>
    <div class="alert alert-danger alert-dismissible fade show">
        <i class="fas fa-exclamation-triangle me-2"></i><?php echo $error; ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
<?php endif; ?>

<!-- Status Summary -->
<div class="row mb-4">
    <div class="col-12">
        <div class="btn-group w-100" role="group">
            <a href="verifications.php" class="btn <?php echo !$status ? 'btn-primary' : 'btn-outline-primary'; ?>">
                Semua (<?php echo formatNumber($totalVerifications); ?>)
            </a>
            <a href="?status=pending" class="btn <?php echo $status === 'pending' ? 'btn-warning' : 'btn-outline-warning'; ?>">
                Pending (<?php echo formatNumber($statusCounts['pending'] ?? 0); ?>)
            </a>
            <a href="?status=processing" class="btn <?php echo $status === 'processing' ? 'btn-info' : 'btn-outline-info'; ?>">
                Processing (<?php echo formatNumber($statusCounts['processing'] ?? 0); ?>)
            </a>
            <a href="?status=success" class="btn <?php echo $status === 'success' ? 'btn-success' : 'btn-outline-success'; ?>">
                Success (<?php echo formatNumber($statusCounts['success'] ?? 0); ?>)
            </a>
            <a href="?status=error" class="btn <?php echo $status === 'error' ? 'btn-danger' : 'btn-outline-danger'; ?>">
                Error (<?php echo formatNumber($statusCounts['error'] ?? 0); ?>)
            </a>
        </div>
    </div>
</div>

<!-- Verifications Table -->
<div class="card">
    <div class="card-header d-flex justify-content-between align-items-center">
        <h5 class="card-title mb-0">
            <i class="fas fa-check-circle me-2"></i>
            Daftar Verifikasi
        </h5>
        <div class="btn-group">
            <button class="btn btn-outline-primary btn-sm" onclick="location.reload()">
                <i class="fas fa-sync-alt me-1"></i>Refresh
            </button>
            <button class="btn btn-outline-success btn-sm" onclick="exportVerifications()">
                <i class="fas fa-download me-1"></i>Export
            </button>
        </div>
    </div>
    <div class="card-body">
        <?php if (empty($verifications)): ?>
            <div class="text-center py-5">
                <i class="fas fa-search fa-3x text-muted mb-3"></i>
                <h5 class="text-muted">Tidak ada verifikasi ditemukan</h5>
                <p class="text-muted">Coba ubah filter pencarian Anda</p>
            </div>
        <?php else: ?>
            <div class="table-responsive">
                <table class="table table-hover">
                    <thead>
                        <tr>
                            <th>User</th>
                            <th>Verification ID</th>
                            <th>Status</th>
                            <th>Check Token</th>
                            <th>Waktu</th>
                            <th>Pesan</th>
                            <th>Aksi</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($verifications as $verification): ?>
                            <tr>
                                <td>
                                    <div class="d-flex align-items-center">
                                        <div class="avatar-sm bg-primary rounded-circle d-flex align-items-center justify-content-center me-2">
                                            <i class="fas fa-user text-white"></i>
                                        </div>
                                        <div>
                                            <div class="fw-bold">
                                                <?php echo htmlspecialchars($verification['first_name'] ?? 'Unknown'); ?>
                                                <?php if ($verification['last_name']): ?>
                                                    <?php echo htmlspecialchars($verification['last_name']); ?>
                                                <?php endif; ?>
                                            </div>
                                            <small class="text-muted">@<?php echo htmlspecialchars($verification['username'] ?? 'N/A'); ?></small>
                                        </div>
                                    </div>
                                </td>
                                <td>
                                    <code><?php echo htmlspecialchars($verification['verification_id']); ?></code>
                                    <button class="btn btn-sm btn-outline-secondary ms-1" 
                                            onclick="copyToClipboard('<?php echo htmlspecialchars($verification['verification_id']); ?>')"
                                            data-bs-toggle="tooltip" title="Copy ID">
                                        <i class="fas fa-copy"></i>
                                    </button>
                                </td>
                                <td>
                                    <span id="status-<?php echo $verification['id']; ?>">
                                        <?php echo getStatusBadge($verification['status']); ?>
                                    </span>
                                </td>
                                <td>
                                    <?php if ($verification['check_token']): ?>
                                        <code class="small"><?php echo substr($verification['check_token'], 0, 10) . '...'; ?></code>
                                    <?php else: ?>
                                        <span class="text-muted">-</span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <div>
                                        <small><?php echo formatDate($verification['created_at']); ?></small>
                                        <?php if ($verification['updated_at'] !== $verification['created_at']): ?>
                                            <br><small class="text-muted">Updated: <?php echo formatDate($verification['updated_at']); ?></small>
                                        <?php endif; ?>
                                    </div>
                                </td>
                                <td>
                                    <small class="text-muted">
                                        <?php echo htmlspecialchars(substr($verification['message'] ?? '', 0, 50)); ?>
                                        <?php if (strlen($verification['message'] ?? '') > 50) echo '...'; ?>
                                    </small>
                                </td>
                                <td>
                                    <div class="btn-group btn-group-sm">
                                        <button class="btn btn-outline-primary btn-action" 
                                                onclick="editVerification('<?php echo $verification['verification_id']; ?>', '<?php echo $verification['status']; ?>', '<?php echo htmlspecialchars($verification['message'] ?? ''); ?>')"
                                                data-bs-toggle="tooltip" title="Edit Status">
                                            <i class="fas fa-edit"></i>
                                        </button>
                                        
                                        <button class="btn btn-outline-danger btn-action" 
                                                onclick="deleteVerification('<?php echo $verification['verification_id']; ?>')"
                                                data-bs-toggle="tooltip" title="Hapus">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            
            <!-- Pagination -->
            <?php if ($totalPages > 1): ?>
                <nav class="mt-4">
                    <ul class="pagination justify-content-center">
                        <?php if ($page > 1): ?>
                            <li class="page-item">
                                <a class="page-link" href="?page=<?php echo $page-1; ?>&<?php echo http_build_query($_GET); ?>">
                                    <i class="fas fa-chevron-left"></i>
                                </a>
                            </li>
                        <?php endif; ?>
                        
                        <?php for ($i = max(1, $page-2); $i <= min($totalPages, $page+2); $i++): ?>
                            <li class="page-item <?php echo $i == $page ? 'active' : ''; ?>">
                                <a class="page-link" href="?page=<?php echo $i; ?>&<?php echo http_build_query(array_merge($_GET, ['page' => $i])); ?>">
                                    <?php echo $i; ?>
                                </a>
                            </li>
                        <?php endfor; ?>
                        
                        <?php if ($page < $totalPages): ?>
                            <li class="page-item">
                                <a class="page-link" href="?page=<?php echo $page+1; ?>&<?php echo http_build_query($_GET); ?>">
                                    <i class="fas fa-chevron-right"></i>
                                </a>
                            </li>
                        <?php endif; ?>
                    </ul>
                </nav>
            <?php endif; ?>
        <?php endif; ?>
    </div>
</div>

<!-- Edit Verification Modal -->
<div class="modal fade" id="editModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Edit Status Verifikasi</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST">
                <div class="modal-body">
                    <input type="hidden" name="action" value="update_status">
                    <input type="hidden" name="verification_id" id="edit_verification_id">
                    
                    <div class="mb-3">
                        <label class="form-label">Status</label>
                        <select class="form-select" name="status" id="edit_status" required>
                            <option value="pending">Pending</option>
                            <option value="processing">Processing</option>
                            <option value="success">Success</option>
                            <option value="error">Error</option>
                            <option value="cancelled">Cancelled</option>
                        </select>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Pesan</label>
                        <textarea class="form-control" name="message" id="edit_message" rows="3"></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                    <button type="submit" class="btn btn-primary">Simpan</button>
                </div>
            </form>
        </div>
    </div>
</div>

<?php
$customJS = "
    function editVerification(verificationId, status, message) {
        document.getElementById('edit_verification_id').value = verificationId;
        document.getElementById('edit_status').value = status;
        document.getElementById('edit_message').value = message;
        
        const modal = new bootstrap.Modal(document.getElementById('editModal'));
        modal.show();
    }
    
    function deleteVerification(verificationId) {
        if (confirm('Apakah Anda yakin ingin menghapus verifikasi ini?')) {
            const form = document.createElement('form');
            form.method = 'POST';
            form.innerHTML = `
                <input type=\"hidden\" name=\"action\" value=\"delete_verification\">
                <input type=\"hidden\" name=\"verification_id\" value=\"${verificationId}\">
            `;
            document.body.appendChild(form);
            form.submit();
        }
    }
    
    function exportVerifications() {
        const params = new URLSearchParams(window.location.search);
        window.open('ajax/export_verifications.php?' + params.toString(), '_blank');
    }
    
    // Auto-refresh every 30 seconds for real-time updates
    setInterval(() => {
        if (document.visibilityState === 'visible') {
            // Only refresh if no modals are open
            if (!document.querySelector('.modal.show')) {
                location.reload();
            }
        }
    }, 30000);
";

include 'includes/footer.php';
?>