<?php
require_once '../config/database.php';
require_once 'config/admin.php';

AdminAuth::requireLogin();

$pageTitle = 'Manajemen Users';
$db = new Database();
$conn = $db->getConnection();

// Handle actions
if ($_POST) {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'delete_user':
                $userId = $_POST['user_id'];
                $stmt = $conn->prepare("DELETE FROM users WHERE id = ?");
                if ($stmt->execute([$userId])) {
                    $success = "User berhasil dihapus!";
                } else {
                    $error = "Gagal menghapus user!";
                }
                break;
                
            case 'reset_tokens':
                $userId = $_POST['user_id'];
                $stmt = $conn->prepare("DELETE FROM user_tokens WHERE user_id = ?");
                if ($stmt->execute([$userId])) {
                    $success = "Token user berhasil direset!";
                } else {
                    $error = "Gagal mereset token!";
                }
                break;
        }
    }
}

// Pagination
$page = $_GET['page'] ?? 1;
$limit = ITEMS_PER_PAGE;
$offset = ($page - 1) * $limit;

// Search
$search = $_GET['search'] ?? '';
$searchCondition = '';
$searchParams = [];

if ($search) {
    $searchCondition = "WHERE u.first_name LIKE ? OR u.username LIKE ? OR u.telegram_id LIKE ?";
    $searchParams = ["%$search%", "%$search%", "%$search%"];
}

// Get total count
$countQuery = "SELECT COUNT(*) as total FROM users u $searchCondition";
$stmt = $conn->prepare($countQuery);
$stmt->execute($searchParams);
$totalUsers = $stmt->fetch()['total'];
$totalPages = ceil($totalUsers / $limit);

// Get users with verification stats
$query = "
    SELECT u.*, 
           COUNT(v.id) as total_verifications,
           SUM(CASE WHEN v.status = 'success' THEN 1 ELSE 0 END) as success_verifications,
           MAX(v.created_at) as last_verification,
           ut.hcaptcha_token IS NOT NULL as has_token
    FROM users u 
    LEFT JOIN verifications v ON u.id = v.user_id 
    LEFT JOIN user_tokens ut ON u.id = ut.user_id
    $searchCondition
    GROUP BY u.id 
    ORDER BY u.created_at DESC 
    LIMIT $limit OFFSET $offset
";

$stmt = $conn->prepare($query);
$stmt->execute($searchParams);
$users = $stmt->fetchAll();

include 'includes/header.php';
?>

<div class="row mb-4">
    <div class="col-md-6">
        <div class="input-group">
            <span class="input-group-text">
                <i class="fas fa-search"></i>
            </span>
            <input type="text" class="form-control" id="searchInput" placeholder="Cari user..." 
                   value="<?php echo htmlspecialchars($search); ?>">
        </div>
    </div>
    <div class="col-md-6 text-end">
        <div class="btn-group">
            <button class="btn btn-outline-primary" onclick="location.reload()">
                <i class="fas fa-sync-alt me-1"></i> Refresh
            </button>
            <button class="btn btn-outline-success" onclick="exportUsers()">
                <i class="fas fa-download me-1"></i> Export
            </button>
        </div>
    </div>
</div>

<?php if (isset($success)): ?>
    <div class="alert alert-success alert-dismissible fade show">
        <i class="fas fa-check-circle me-2"></i><?php echo $success; ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
<?php endif; ?>

<?php if (isset($error)): ?>
    <div class="alert alert-danger alert-dismissible fade show">
        <i class="fas fa-exclamation-triangle me-2"></i><?php echo $error; ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
<?php endif; ?>

<!-- Statistics Cards -->
<div class="row mb-4">
    <div class="col-md-3">
        <div class="card text-center">
            <div class="card-body">
                <h3 class="text-primary"><?php echo formatNumber($totalUsers); ?></h3>
                <p class="text-muted mb-0">Total Users</p>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="card text-center">
            <div class="card-body">
                <?php
                $stmt = $conn->query("SELECT COUNT(*) as total FROM users WHERE created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)");
                $newUsers = $stmt->fetch()['total'];
                ?>
                <h3 class="text-success"><?php echo formatNumber($newUsers); ?></h3>
                <p class="text-muted mb-0">User Baru (7 hari)</p>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="card text-center">
            <div class="card-body">
                <?php
                $stmt = $conn->query("SELECT COUNT(DISTINCT user_id) as total FROM user_tokens");
                $usersWithTokens = $stmt->fetch()['total'];
                ?>
                <h3 class="text-info"><?php echo formatNumber($usersWithTokens); ?></h3>
                <p class="text-muted mb-0">Punya Token</p>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="card text-center">
            <div class="card-body">
                <?php
                $stmt = $conn->query("SELECT COUNT(DISTINCT user_id) as total FROM verifications WHERE created_at >= CURDATE()");
                $activeToday = $stmt->fetch()['total'];
                ?>
                <h3 class="text-warning"><?php echo formatNumber($activeToday); ?></h3>
                <p class="text-muted mb-0">Aktif Hari Ini</p>
            </div>
        </div>
    </div>
</div>

<!-- Users Table -->
<div class="card">
    <div class="card-header">
        <h5 class="card-title mb-0">
            <i class="fas fa-users me-2"></i>
            Daftar Users
        </h5>
    </div>
    <div class="card-body">
        <?php if (empty($users)): ?>
            <div class="text-center py-5">
                <i class="fas fa-users fa-3x text-muted mb-3"></i>
                <h5 class="text-muted">Tidak ada user ditemukan</h5>
                <?php if ($search): ?>
                    <p class="text-muted">Coba ubah kata kunci pencarian</p>
                    <a href="users.php" class="btn btn-outline-primary">Reset Pencarian</a>
                <?php endif; ?>
            </div>
        <?php else: ?>
            <div class="table-responsive">
                <table class="table table-hover" id="usersTable">
                    <thead>
                        <tr>
                            <th>User</th>
                            <th>Telegram ID</th>
                            <th>Verifikasi</th>
                            <th>Token</th>
                            <th>Terakhir Aktif</th>
                            <th>Bergabung</th>
                            <th>Aksi</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($users as $user): ?>
                            <tr>
                                <td>
                                    <div class="d-flex align-items-center">
                                        <div class="avatar-sm bg-primary rounded-circle d-flex align-items-center justify-content-center me-3">
                                            <i class="fas fa-user text-white"></i>
                                        </div>
                                        <div>
                                            <div class="fw-bold">
                                                <?php echo htmlspecialchars($user['first_name'] ?? 'Unknown'); ?>
                                                <?php if ($user['last_name']): ?>
                                                    <?php echo htmlspecialchars($user['last_name']); ?>
                                                <?php endif; ?>
                                            </div>
                                            <small class="text-muted">
                                                @<?php echo htmlspecialchars($user['username'] ?? 'N/A'); ?>
                                            </small>
                                        </div>
                                    </div>
                                </td>
                                <td>
                                    <code><?php echo $user['telegram_id']; ?></code>
                                </td>
                                <td>
                                    <div>
                                        <span class="badge bg-primary"><?php echo $user['total_verifications']; ?> Total</span>
                                        <?php if ($user['success_verifications'] > 0): ?>
                                            <span class="badge bg-success"><?php echo $user['success_verifications']; ?> Sukses</span>
                                        <?php endif; ?>
                                    </div>
                                </td>
                                <td>
                                    <?php if ($user['has_token']): ?>
                                        <span class="badge bg-success">
                                            <i class="fas fa-check me-1"></i>Ada
                                        </span>
                                    <?php else: ?>
                                        <span class="badge bg-secondary">
                                            <i class="fas fa-times me-1"></i>Tidak Ada
                                        </span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php if ($user['last_verification']): ?>
                                        <small><?php echo formatDate($user['last_verification']); ?></small>
                                    <?php else: ?>
                                        <small class="text-muted">Belum pernah</small>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <small><?php echo formatDate($user['created_at']); ?></small>
                                </td>
                                <td>
                                    <div class="btn-group btn-group-sm">
                                        <button class="btn btn-outline-info btn-action" 
                                                onclick="viewUser(<?php echo $user['id']; ?>)"
                                                data-bs-toggle="tooltip" title="Lihat Detail">
                                            <i class="fas fa-eye"></i>
                                        </button>
                                        
                                        <?php if ($user['has_token']): ?>
                                            <button class="btn btn-outline-warning btn-action" 
                                                    onclick="resetTokens(<?php echo $user['id']; ?>)"
                                                    data-bs-toggle="tooltip" title="Reset Token">
                                                <i class="fas fa-key"></i>
                                            </button>
                                        <?php endif; ?>
                                        
                                        <button class="btn btn-outline-danger btn-action" 
                                                onclick="deleteUser(<?php echo $user['id']; ?>)"
                                                data-bs-toggle="tooltip" title="Hapus User">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            
            <!-- Pagination -->
            <?php if ($totalPages > 1): ?>
                <nav class="mt-4">
                    <ul class="pagination justify-content-center">
                        <?php if ($page > 1): ?>
                            <li class="page-item">
                                <a class="page-link" href="?page=<?php echo $page-1; ?><?php echo $search ? '&search='.urlencode($search) : ''; ?>">
                                    <i class="fas fa-chevron-left"></i>
                                </a>
                            </li>
                        <?php endif; ?>
                        
                        <?php for ($i = max(1, $page-2); $i <= min($totalPages, $page+2); $i++): ?>
                            <li class="page-item <?php echo $i == $page ? 'active' : ''; ?>">
                                <a class="page-link" href="?page=<?php echo $i; ?><?php echo $search ? '&search='.urlencode($search) : ''; ?>">
                                    <?php echo $i; ?>
                                </a>
                            </li>
                        <?php endfor; ?>
                        
                        <?php if ($page < $totalPages): ?>
                            <li class="page-item">
                                <a class="page-link" href="?page=<?php echo $page+1; ?><?php echo $search ? '&search='.urlencode($search) : ''; ?>">
                                    <i class="fas fa-chevron-right"></i>
                                </a>
                            </li>
                        <?php endif; ?>
                    </ul>
                </nav>
            <?php endif; ?>
        <?php endif; ?>
    </div>
</div>

<!-- User Detail Modal -->
<div class="modal fade" id="userModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Detail User</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body" id="userModalBody">
                <div class="text-center">
                    <div class="spinner-border" role="status">
                        <span class="visually-hidden">Loading...</span>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<?php
$customJS = "
    // Search functionality
    document.getElementById('searchInput').addEventListener('keypress', function(e) {
        if (e.key === 'Enter') {
            const search = this.value;
            const url = new URL(window.location);
            if (search) {
                url.searchParams.set('search', search);
            } else {
                url.searchParams.delete('search');
            }
            url.searchParams.delete('page');
            window.location = url;
        }
    });
    
    function viewUser(userId) {
        const modal = new bootstrap.Modal(document.getElementById('userModal'));
        modal.show();
        
        // Load user details via AJAX
        fetch('ajax/user_details.php?id=' + userId)
            .then(response => response.text())
            .then(html => {
                document.getElementById('userModalBody').innerHTML = html;
            })
            .catch(error => {
                document.getElementById('userModalBody').innerHTML = 
                    '<div class=\"alert alert-danger\">Gagal memuat data user</div>';
            });
    }
    
    function deleteUser(userId) {
        if (confirm('Apakah Anda yakin ingin menghapus user ini? Semua data verifikasi akan ikut terhapus.')) {
            const form = document.createElement('form');
            form.method = 'POST';
            form.innerHTML = `
                <input type=\"hidden\" name=\"action\" value=\"delete_user\">
                <input type=\"hidden\" name=\"user_id\" value=\"${userId}\">
            `;
            document.body.appendChild(form);
            form.submit();
        }
    }
    
    function resetTokens(userId) {
        if (confirm('Apakah Anda yakin ingin mereset token user ini?')) {
            const form = document.createElement('form');
            form.method = 'POST';
            form.innerHTML = `
                <input type=\"hidden\" name=\"action\" value=\"reset_tokens\">
                <input type=\"hidden\" name=\"user_id\" value=\"${userId}\">
            `;
            document.body.appendChild(form);
            form.submit();
        }
    }
    
    function exportUsers() {
        window.open('ajax/export_users.php', '_blank');
    }
";

include 'includes/footer.php';
?>