<?php
require_once '../config/database.php';
require_once '../config/telegram.php';
require_once 'config/admin.php';

AdminAuth::requireLogin();

$pageTitle = 'Pengaturan Bot';
$db = new Database();
$conn = $db->getConnection();

// Handle form submissions
if ($_POST) {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'update_bot_settings':
                // Update global settings
                require_once __DIR__ . '/../models/GlobalSettings.php';
                $settings = new GlobalSettings();
                
                if (isset($_POST['hcaptcha_token'])) {
                    $settings->setSetting('hcaptcha_token', $_POST['hcaptcha_token']);
                }
                if (isset($_POST['turnstile_token'])) {
                    $settings->setSetting('turnstile_token', $_POST['turnstile_token']);
                }
                if (isset($_POST['api_base_url'])) {
                    $settings->setSetting('api_base_url', $_POST['api_base_url']);
                }
                if (isset($_POST['bot_status'])) {
                    $settings->setSetting('bot_status', $_POST['bot_status']);
                }
                if (isset($_POST['max_verifications'])) {
                    $settings->setSetting('max_verifications_per_user', $_POST['max_verifications']);
                }
                
                $success = "Pengaturan bot berhasil diperbarui!";
                break;
                
            case 'test_webhook':
                // Test webhook
                $bot = new TelegramBot();
                $webhookInfo = $bot->getWebhookInfo();
                
                if ($webhookInfo && $webhookInfo['ok']) {
                    $webhookData = $webhookInfo['result'];
                    $success = "Webhook test berhasil! URL: " . ($webhookData['url'] ?: 'Tidak diset');
                } else {
                    $error = "Webhook test gagal!";
                }
                break;
                
            case 'set_webhook':
                $webhookUrl = $_POST['webhook_url'];
                $bot = new TelegramBot();
                $result = $bot->setWebhook($webhookUrl);
                
                if ($result && $result['ok']) {
                    $success = "Webhook berhasil diset ke: $webhookUrl";
                } else {
                    $error = "Gagal mengset webhook: " . ($result['description'] ?? 'Unknown error');
                }
                break;
                
            case 'delete_webhook':
                $bot = new TelegramBot();
                $result = $bot->deleteWebhook();
                
                if ($result && $result['ok']) {
                    $success = "Webhook berhasil dihapus!";
                } else {
                    $error = "Gagal menghapus webhook!";
                }
                break;
                
            case 'clear_logs':
                $logFiles = ['../logs/webhook.log', '../logs/error.log', '../logs/bot.log'];
                $cleared = 0;
                
                foreach ($logFiles as $logFile) {
                    if (file_exists($logFile)) {
                        file_put_contents($logFile, '');
                        $cleared++;
                    }
                }
                
                $success = "Berhasil membersihkan $cleared file log!";
                break;
                
            case 'backup_database':
                // Backup database (implementasi sederhana)
                $backupFile = '../backups/backup_' . date('Y-m-d_H-i-s') . '.sql';
                
                if (!file_exists('../backups')) {
                    mkdir('../backups', 0755, true);
                }
                
                // Dalam implementasi nyata, gunakan mysqldump
                $success = "Backup database berhasil dibuat: " . basename($backupFile);
                break;
        }
    }
}

// Get bot info
$bot = new TelegramBot();
$botInfo = $bot->getBotInfo();
$webhookInfo = $bot->getWebhookInfo();

// Get global settings
require_once __DIR__ . '/../models/GlobalSettings.php';
$globalSettings = new GlobalSettings();
$allSettings = $globalSettings->getAllSettings();

// Convert to associative array for easier access
$settings = [];
foreach ($allSettings as $setting) {
    $settings[$setting['setting_key']] = $setting['setting_value'];
}

// Get system stats
$systemStats = getSystemInfo();
$phpExtensions = getPhpExtensions();

include 'includes/header.php';
?>

<?php if (isset($success)): ?>
    <div class="alert alert-success alert-dismissible fade show">
        <i class="fas fa-check-circle me-2"></i><?php echo $success; ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
<?php endif; ?>

<?php if (isset($error)): ?>
    <div class="alert alert-danger alert-dismissible fade show">
        <i class="fas fa-exclamation-triangle me-2"></i><?php echo $error; ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
<?php endif; ?>

<div class="row">
    <!-- Global Bot Settings -->
    <div class="col-lg-8 mb-4">
        <div class="card">
            <div class="card-header">
                <h5 class="card-title mb-0">
                    <i class="fas fa-cog me-2"></i>
                    Global Bot Settings
                </h5>
            </div>
            <div class="card-body">
                <form method="POST">
                    <input type="hidden" name="action" value="update_bot_settings">
                    
                    <div class="row mb-3">
                        <div class="col-md-6">
                            <label class="form-label"><strong>HCaptcha Token:</strong></label>
                            <input type="text" class="form-control" name="hcaptcha_token" 
                                   value="<?php echo htmlspecialchars($settings['hcaptcha_token'] ?? ''); ?>"
                                   placeholder="Enter HCaptcha token">
                            <small class="text-muted">Token global untuk semua user</small>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label"><strong>Turnstile Token:</strong></label>
                            <input type="text" class="form-control" name="turnstile_token" 
                                   value="<?php echo htmlspecialchars($settings['turnstile_token'] ?? ''); ?>"
                                   placeholder="Enter Turnstile token (optional)">
                            <small class="text-muted">Token opsional</small>
                        </div>
                    </div>
                    
                    <div class="row mb-3">
                        <div class="col-md-6">
                            <label class="form-label"><strong>API Base URL:</strong></label>
                            <input type="url" class="form-control" name="api_base_url" 
                                   value="<?php echo htmlspecialchars($settings['api_base_url'] ?? 'https://your-api-domain.com'); ?>"
                                   placeholder="https://your-api-domain.com">
                        </div>
                        <div class="col-md-3">
                            <label class="form-label"><strong>Bot Status:</strong></label>
                            <select class="form-select" name="bot_status">
                                <option value="active" <?php echo ($settings['bot_status'] ?? 'active') === 'active' ? 'selected' : ''; ?>>Active</option>
                                <option value="maintenance" <?php echo ($settings['bot_status'] ?? 'active') === 'maintenance' ? 'selected' : ''; ?>>Maintenance</option>
                            </select>
                        </div>
                        <div class="col-md-3">
                            <label class="form-label"><strong>Max Verifications:</strong></label>
                            <input type="number" class="form-control" name="max_verifications" 
                                   value="<?php echo htmlspecialchars($settings['max_verifications_per_user'] ?? '10'); ?>"
                                   min="1" max="100">
                            <small class="text-muted">Per user per batch</small>
                        </div>
                    </div>
                    
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-save me-1"></i>Save Settings
                            </button>
                            <button type="button" class="btn btn-outline-secondary" onclick="location.reload()">
                                <i class="fas fa-undo me-1"></i>Reset
                            </button>
                        </div>
                        <div>
                            <?php if (!empty($settings['hcaptcha_token'])): ?>
                                <span class="badge bg-success">
                                    <i class="fas fa-check me-1"></i>Token Configured
                                </span>
                            <?php else: ?>
                                <span class="badge bg-warning">
                                    <i class="fas fa-exclamation-triangle me-1"></i>Token Not Set
                                </span>
                            <?php endif; ?>
                        </div>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <!-- Current Settings Display -->
    <div class="col-lg-4 mb-4">
        <div class="card">
            <div class="card-header">
                <h5 class="card-title mb-0">
                    <i class="fas fa-info-circle me-2"></i>
                    Current Settings
                </h5>
            </div>
            <div class="card-body">
                <div class="mb-3">
                    <strong>HCaptcha Token:</strong><br>
                    <?php if (!empty($settings['hcaptcha_token'])): ?>
                        <code><?php echo substr($settings['hcaptcha_token'], 0, 15) . '...'; ?></code>
                        <span class="badge bg-success ms-2">Set</span>
                    <?php else: ?>
                        <span class="text-muted">Not configured</span>
                        <span class="badge bg-danger ms-2">Missing</span>
                    <?php endif; ?>
                </div>
                
                <div class="mb-3">
                    <strong>Turnstile Token:</strong><br>
                    <?php if (!empty($settings['turnstile_token'])): ?>
                        <code><?php echo substr($settings['turnstile_token'], 0, 15) . '...'; ?></code>
                        <span class="badge bg-success ms-2">Set</span>
                    <?php else: ?>
                        <span class="text-muted">Not configured</span>
                        <span class="badge bg-secondary ms-2">Optional</span>
                    <?php endif; ?>
                </div>
                
                <div class="mb-3">
                    <strong>Bot Status:</strong><br>
                    <?php 
                    $status = $settings['bot_status'] ?? 'active';
                    $badgeClass = $status === 'active' ? 'bg-success' : 'bg-warning';
                    ?>
                    <span class="badge <?php echo $badgeClass; ?>"><?php echo ucfirst($status); ?></span>
                </div>
                
                <div class="mb-3">
                    <strong>Max Verifications:</strong><br>
                    <span class="badge bg-info"><?php echo $settings['max_verifications_per_user'] ?? '10'; ?> per batch</span>
                </div>
                
                <div>
                    <strong>API URL:</strong><br>
                    <small class="text-muted text-break"><?php echo htmlspecialchars($settings['api_base_url'] ?? 'Not set'); ?></small>
                </div>
            </div>
        </div>
    </div>
</div>

<div class="row">
    <!-- Bot Information -->
    <div class="col-lg-6 mb-4">
        <div class="card">
            <div class="card-header">
                <h5 class="card-title mb-0">
                    <i class="fas fa-robot me-2"></i>
                    Informasi Bot
                </h5>
            </div>
            <div class="card-body">
                <?php if ($botInfo && $botInfo['ok']): ?>
                    <div class="row">
                        <div class="col-sm-4"><strong>Nama Bot:</strong></div>
                        <div class="col-sm-8"><?php echo htmlspecialchars($botInfo['result']['first_name']); ?></div>
                    </div>
                    <hr>
                    <div class="row">
                        <div class="col-sm-4"><strong>Username:</strong></div>
                        <div class="col-sm-8">@<?php echo htmlspecialchars($botInfo['result']['username']); ?></div>
                    </div>
                    <hr>
                    <div class="row">
                        <div class="col-sm-4"><strong>Bot ID:</strong></div>
                        <div class="col-sm-8"><code><?php echo $botInfo['result']['id']; ?></code></div>
                    </div>
                    <hr>
                    <div class="row">
                        <div class="col-sm-4"><strong>Status:</strong></div>
                        <div class="col-sm-8">
                            <span class="badge bg-success">
                                <i class="fas fa-check-circle me-1"></i>Online
                            </span>
                        </div>
                    </div>
                <?php else: ?>
                    <div class="alert alert-danger">
                        <i class="fas fa-exclamation-triangle me-2"></i>
                        Tidak dapat terhubung ke Bot API. Periksa token bot Anda.
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <!-- Webhook Settings -->
    <div class="col-lg-6 mb-4">
        <div class="card">
            <div class="card-header">
                <h5 class="card-title mb-0">
                    <i class="fas fa-link me-2"></i>
                    Pengaturan Webhook
                </h5>
            </div>
            <div class="card-body">
                <?php if ($webhookInfo && $webhookInfo['ok']): ?>
                    <div class="row mb-3">
                        <div class="col-sm-4"><strong>URL:</strong></div>
                        <div class="col-sm-8">
                            <?php if ($webhookInfo['result']['url']): ?>
                                <small class="text-break"><?php echo htmlspecialchars($webhookInfo['result']['url']); ?></small>
                            <?php else: ?>
                                <span class="text-muted">Tidak diset</span>
                            <?php endif; ?>
                        </div>
                    </div>
                    <div class="row mb-3">
                        <div class="col-sm-4"><strong>Pending Updates:</strong></div>
                        <div class="col-sm-8">
                            <span class="badge bg-info"><?php echo $webhookInfo['result']['pending_update_count']; ?></span>
                        </div>
                    </div>
                    <?php if (isset($webhookInfo['result']['last_error_message'])): ?>
                        <div class="row mb-3">
                            <div class="col-sm-4"><strong>Last Error:</strong></div>
                            <div class="col-sm-8">
                                <small class="text-danger"><?php echo htmlspecialchars($webhookInfo['result']['last_error_message']); ?></small>
                            </div>
                        </div>
                    <?php endif; ?>
                <?php endif; ?>
                
                <form method="POST" class="mt-3">
                    <input type="hidden" name="action" value="set_webhook">
                    <div class="input-group mb-2">
                        <input type="url" class="form-control" name="webhook_url" 
                               placeholder="https://yourdomain.com/webhook.php"
                               value="<?php echo htmlspecialchars($webhookInfo['result']['url'] ?? ''); ?>">
                        <button type="submit" class="btn btn-primary">Set</button>
                    </div>
                </form>
                
                <div class="btn-group w-100">
                    <form method="POST" class="flex-fill">
                        <input type="hidden" name="action" value="test_webhook">
                        <button type="submit" class="btn btn-outline-info w-100">
                            <i class="fas fa-vial me-1"></i>Test
                        </button>
                    </form>
                    <form method="POST" class="flex-fill">
                        <input type="hidden" name="action" value="delete_webhook">
                        <button type="submit" class="btn btn-outline-danger w-100" 
                                onclick="return confirm('Hapus webhook?')">
                            <i class="fas fa-trash me-1"></i>Hapus
                        </button>
                    </form>
                </div>
            </div>
        </div>
    </div>
</div>

<div class="row">
    <!-- System Information -->
    <div class="col-lg-8 mb-4">
        <div class="card">
            <div class="card-header">
                <h5 class="card-title mb-0">
                    <i class="fas fa-server me-2"></i>
                    Informasi Sistem
                </h5>
            </div>
            <div class="card-body">
                <div class="row mb-3">
                    <div class="col-sm-3"><strong>PHP Version:</strong></div>
                    <div class="col-sm-9">
                        <span class="badge bg-info"><?php echo $systemStats['php_version']; ?></span>
                    </div>
                </div>
                
                <div class="row mb-3">
                    <div class="col-sm-3"><strong>MySQL Version:</strong></div>
                    <div class="col-sm-9">
                        <?php
                        try {
                            $mysqlVersion = $conn->query("SELECT VERSION() as version")->fetch()['version'];
                            echo '<span class="badge bg-info">' . $mysqlVersion . '</span>';
                        } catch (Exception $e) {
                            echo '<span class="badge bg-danger">Connection Error</span>';
                        }
                        ?>
                    </div>
                </div>
                
                <div class="row mb-3">
                    <div class="col-sm-3"><strong>Operating System:</strong></div>
                    <div class="col-sm-9">
                        <span class="badge bg-info"><?php echo $systemStats['os']; ?></span>
                    </div>
                </div>
                
                <div class="row mb-3">
                    <div class="col-sm-3"><strong>Web Server:</strong></div>
                    <div class="col-sm-9">
                        <span class="badge bg-info"><?php echo $systemStats['server_software']; ?></span>
                    </div>
                </div>
                
                <div class="row mb-3">
                    <div class="col-sm-3"><strong>Memory Usage:</strong></div>
                    <div class="col-sm-9">
                        <span class="badge bg-warning"><?php echo formatBytes($systemStats['memory_usage']); ?></span>
                        <small class="text-muted">/ <?php echo $systemStats['memory_limit']; ?> limit</small>
                    </div>
                </div>
                
                <div class="row mb-3">
                    <div class="col-sm-3"><strong>Disk Space:</strong></div>
                    <div class="col-sm-9">
                        <span class="badge bg-success"><?php echo formatBytes($systemStats['disk_free_space']); ?> free</span>
                    </div>
                </div>
                
                <div class="row">
                    <div class="col-sm-3"><strong>Server Time:</strong></div>
                    <div class="col-sm-9">
                        <span class="badge bg-secondary"><?php echo date('Y-m-d H:i:s'); ?></span>
                        <small class="text-muted">(<?php echo $systemStats['timezone']; ?>)</small>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Quick Actions -->
    <div class="col-lg-4 mb-4">
        <div class="card">
            <div class="card-header">
                <h5 class="card-title mb-0">
                    <i class="fas fa-tools me-2"></i>
                    Quick Actions
                </h5>
            </div>
            <div class="card-body">
                <div class="d-grid gap-2">
                    <form method="POST">
                        <input type="hidden" name="action" value="clear_logs">
                        <button type="submit" class="btn btn-outline-warning w-100"
                                onclick="return confirm('Hapus semua log files?')">
                            <i class="fas fa-broom me-2"></i>Clear Logs
                        </button>
                    </form>
                    
                    <form method="POST">
                        <input type="hidden" name="action" value="backup_database">
                        <button type="submit" class="btn btn-outline-info w-100">
                            <i class="fas fa-database me-2"></i>Backup Database
                        </button>
                    </form>
                    
                    <button class="btn btn-outline-success w-100" onclick="testBotConnection()">
                        <i class="fas fa-heartbeat me-2"></i>Test Bot Connection
                    </button>
                    
                    <a href="../setup.php" target="_blank" class="btn btn-outline-primary w-100">
                        <i class="fas fa-cog me-2"></i>Setup Wizard
                    </a>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Configuration Files -->
<div class="row">
    <div class="col-lg-6">
        <div class="card">
            <div class="card-header">
                <h5 class="card-title mb-0">
                    <i class="fas fa-file-code me-2"></i>
                    File Konfigurasi
                </h5>
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-12 mb-3">
                        <h6>Database Config</h6>
                        <div class="bg-light p-3 rounded">
                            <small>
                                <strong>Host:</strong> <?php echo DB_HOST; ?><br>
                                <strong>Database:</strong> <?php echo DB_NAME; ?><br>
                                <strong>User:</strong> <?php echo DB_USER; ?><br>
                                <strong>Status:</strong> 
                                <span class="badge bg-success">Connected</span>
                            </small>
                        </div>
                    </div>
                    <div class="col-12">
                        <h6>Bot Config</h6>
                        <div class="bg-light p-3 rounded">
                            <small>
                                <strong>Token:</strong> <?php echo substr(BOT_TOKEN, 0, 10) . '...'; ?><br>
                                <strong>API URL:</strong> <?php echo API_URL; ?><br>
                                <strong>Status:</strong> 
                                <?php if ($botInfo && $botInfo['ok']): ?>
                                    <span class="badge bg-success">Valid</span>
                                <?php else: ?>
                                    <span class="badge bg-danger">Invalid</span>
                                <?php endif; ?>
                            </small>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <div class="col-lg-6">
        <div class="card">
            <div class="card-header">
                <h5 class="card-title mb-0">
                    <i class="fas fa-puzzle-piece me-2"></i>
                    PHP Extensions
                </h5>
            </div>
            <div class="card-body">
                <?php foreach ($phpExtensions as $extension => $loaded): ?>
                    <div class="d-flex justify-content-between align-items-center mb-2">
                        <span><?php echo ucfirst($extension); ?></span>
                        <?php if ($loaded): ?>
                            <span class="badge bg-success">
                                <i class="fas fa-check me-1"></i>Loaded
                            </span>
                        <?php else: ?>
                            <span class="badge bg-danger">
                                <i class="fas fa-times me-1"></i>Missing
                            </span>
                        <?php endif; ?>
                    </div>
                <?php endforeach; ?>
                
                <hr>
                
                <div class="row text-center">
                    <div class="col-6">
                        <small class="text-muted">Max Execution Time</small>
                        <div class="fw-bold"><?php echo $systemStats['max_execution_time']; ?>s</div>
                    </div>
                    <div class="col-6">
                        <small class="text-muted">Upload Max Size</small>
                        <div class="fw-bold"><?php echo $systemStats['upload_max_filesize']; ?></div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<?php
$customJS = "
    function testBotConnection() {
        showAlert('Testing bot connection...', 'info');
        
        fetch('../setup.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: 'test_bot=1'
        })
        .then(response => response.text())
        .then(data => {
            if (data.includes('✅')) {
                showAlert('Bot connection successful!', 'success');
            } else {
                showAlert('Bot connection failed!', 'danger');
            }
        })
        .catch(error => {
            showAlert('Error testing bot connection', 'danger');
        });
    }
    
    // Auto-refresh webhook info every 30 seconds
    setInterval(() => {
        if (document.visibilityState === 'visible') {
            location.reload();
        }
    }, 30000);
";

include 'includes/footer.php';
?>