<?php
require_once '../config/database.php';
require_once 'config/admin.php';

AdminAuth::requireLogin();

$pageTitle = 'System Logs';

// Handle actions
if ($_POST) {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'clear_log':
                $logFile = $_POST['log_file'];
                $allowedLogs = ['webhook.log', 'error.log', 'bot.log'];
                
                if (in_array($logFile, $allowedLogs)) {
                    $filePath = "../logs/$logFile";
                    if (file_exists($filePath)) {
                        file_put_contents($filePath, '');
                        $success = "Log file $logFile berhasil dibersihkan!";
                    } else {
                        $error = "Log file $logFile tidak ditemukan!";
                    }
                } else {
                    $error = "Log file tidak valid!";
                }
                break;
        }
    }
}

// Get log files
$logDir = '../logs/';
$logFiles = [];

if (is_dir($logDir)) {
    $files = scandir($logDir);
    foreach ($files as $file) {
        if ($file != '.' && $file != '..' && pathinfo($file, PATHINFO_EXTENSION) == 'log') {
            $filePath = $logDir . $file;
            $logFiles[] = [
                'name' => $file,
                'size' => file_exists($filePath) ? filesize($filePath) : 0,
                'modified' => file_exists($filePath) ? filemtime($filePath) : 0,
                'lines' => file_exists($filePath) ? count(file($filePath)) : 0
            ];
        }
    }
}

// Get selected log content
$selectedLog = $_GET['log'] ?? '';
$logContent = '';
$maxLines = 100;

if ($selectedLog && in_array($selectedLog, array_column($logFiles, 'name'))) {
    $filePath = $logDir . $selectedLog;
    if (file_exists($filePath)) {
        $lines = file($filePath);
        $totalLines = count($lines);
        
        // Get last N lines
        $lines = array_slice($lines, -$maxLines);
        $logContent = implode('', $lines);
        
        if ($totalLines > $maxLines) {
            $logContent = "... (showing last $maxLines of $totalLines lines)\n\n" . $logContent;
        }
    }
}

include 'includes/header.php';
?>

<?php if (isset($success)): ?>
    <div class="alert alert-success alert-dismissible fade show">
        <i class="fas fa-check-circle me-2"></i><?php echo $success; ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
<?php endif; ?>

<?php if (isset($error)): ?>
    <div class="alert alert-danger alert-dismissible fade show">
        <i class="fas fa-exclamation-triangle me-2"></i><?php echo $error; ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
<?php endif; ?>

<div class="row">
    <!-- Log Files List -->
    <div class="col-lg-4 mb-4">
        <div class="card">
            <div class="card-header">
                <h5 class="card-title mb-0">
                    <i class="fas fa-file-alt me-2"></i>
                    Log Files
                </h5>
            </div>
            <div class="card-body">
                <?php if (empty($logFiles)): ?>
                    <div class="text-center py-3">
                        <i class="fas fa-folder-open fa-2x text-muted mb-2"></i>
                        <p class="text-muted mb-0">Tidak ada log files</p>
                    </div>
                <?php else: ?>
                    <div class="list-group list-group-flush">
                        <?php foreach ($logFiles as $logFile): ?>
                            <div class="list-group-item <?php echo $selectedLog === $logFile['name'] ? 'active' : ''; ?>">
                                <div class="d-flex justify-content-between align-items-start">
                                    <div class="flex-grow-1">
                                        <a href="?log=<?php echo urlencode($logFile['name']); ?>" 
                                           class="text-decoration-none <?php echo $selectedLog === $logFile['name'] ? 'text-white' : ''; ?>">
                                            <h6 class="mb-1"><?php echo htmlspecialchars($logFile['name']); ?></h6>
                                        </a>
                                        <small class="<?php echo $selectedLog === $logFile['name'] ? 'text-white-50' : 'text-muted'; ?>">
                                            <?php echo number_format($logFile['size'] / 1024, 1); ?> KB • 
                                            <?php echo $logFile['lines']; ?> lines
                                        </small>
                                        <br>
                                        <small class="<?php echo $selectedLog === $logFile['name'] ? 'text-white-50' : 'text-muted'; ?>">
                                            Modified: <?php echo date('d/m/Y H:i', $logFile['modified']); ?>
                                        </small>
                                    </div>
                                    <div class="btn-group btn-group-sm">
                                        <a href="?log=<?php echo urlencode($logFile['name']); ?>" 
                                           class="btn btn-outline-primary btn-sm">
                                            <i class="fas fa-eye"></i>
                                        </a>
                                        <button class="btn btn-outline-danger btn-sm" 
                                                onclick="clearLog('<?php echo htmlspecialchars($logFile['name']); ?>')">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        
        <!-- Log Actions -->
        <div class="card mt-3">
            <div class="card-header">
                <h5 class="card-title mb-0">
                    <i class="fas fa-tools me-2"></i>
                    Log Actions
                </h5>
            </div>
            <div class="card-body">
                <div class="d-grid gap-2">
                    <button class="btn btn-outline-primary" onclick="location.reload()">
                        <i class="fas fa-sync-alt me-2"></i>Refresh Logs
                    </button>
                    
                    <button class="btn btn-outline-warning" onclick="clearAllLogs()">
                        <i class="fas fa-broom me-2"></i>Clear All Logs
                    </button>
                    
                    <?php if ($selectedLog): ?>
                        <a href="ajax/download_log.php?log=<?php echo urlencode($selectedLog); ?>" 
                           class="btn btn-outline-success">
                            <i class="fas fa-download me-2"></i>Download Log
                        </a>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Log Content -->
    <div class="col-lg-8">
        <div class="card">
            <div class="card-header d-flex justify-content-between align-items-center">
                <h5 class="card-title mb-0">
                    <i class="fas fa-terminal me-2"></i>
                    <?php echo $selectedLog ? htmlspecialchars($selectedLog) : 'Select a log file'; ?>
                </h5>
                <?php if ($selectedLog): ?>
                    <div class="btn-group btn-group-sm">
                        <button class="btn btn-outline-primary" onclick="location.reload()">
                            <i class="fas fa-sync-alt me-1"></i>Refresh
                        </button>
                        <button class="btn btn-outline-secondary" onclick="toggleWrap()">
                            <i class="fas fa-text-width me-1"></i>Wrap
                        </button>
                    </div>
                <?php endif; ?>
            </div>
            <div class="card-body p-0">
                <?php if (!$selectedLog): ?>
                    <div class="text-center py-5">
                        <i class="fas fa-file-alt fa-3x text-muted mb-3"></i>
                        <h5 class="text-muted">Pilih log file untuk melihat isinya</h5>
                        <p class="text-muted">Klik pada salah satu log file di sebelah kiri</p>
                    </div>
                <?php elseif (empty($logContent)): ?>
                    <div class="text-center py-5">
                        <i class="fas fa-file fa-3x text-muted mb-3"></i>
                        <h5 class="text-muted">Log file kosong</h5>
                        <p class="text-muted">Belum ada log yang tercatat</p>
                    </div>
                <?php else: ?>
                    <pre id="logContent" class="mb-0 p-3" style="height: 600px; overflow-y: auto; font-size: 12px; line-height: 1.4; white-space: pre-wrap;"><?php echo htmlspecialchars($logContent); ?></pre>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- Log Statistics -->
<?php if (!empty($logFiles)): ?>
<div class="row mt-4">
    <div class="col-12">
        <div class="card">
            <div class="card-header">
                <h5 class="card-title mb-0">
                    <i class="fas fa-chart-bar me-2"></i>
                    Log Statistics
                </h5>
            </div>
            <div class="card-body">
                <div class="row">
                    <?php foreach ($logFiles as $logFile): ?>
                        <div class="col-md-3 mb-3">
                            <div class="card bg-light">
                                <div class="card-body text-center">
                                    <h6 class="card-title"><?php echo htmlspecialchars($logFile['name']); ?></h6>
                                    <div class="row">
                                        <div class="col-6">
                                            <small class="text-muted">Size</small>
                                            <div class="fw-bold"><?php echo number_format($logFile['size'] / 1024, 1); ?> KB</div>
                                        </div>
                                        <div class="col-6">
                                            <small class="text-muted">Lines</small>
                                            <div class="fw-bold"><?php echo number_format($logFile['lines']); ?></div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
    </div>
</div>
<?php endif; ?>

<?php
$customJS = "
    function clearLog(logFile) {
        if (confirm('Apakah Anda yakin ingin menghapus isi log file: ' + logFile + '?')) {
            const form = document.createElement('form');
            form.method = 'POST';
            form.innerHTML = `
                <input type=\"hidden\" name=\"action\" value=\"clear_log\">
                <input type=\"hidden\" name=\"log_file\" value=\"${logFile}\">
            `;
            document.body.appendChild(form);
            form.submit();
        }
    }
    
    function clearAllLogs() {
        if (confirm('Apakah Anda yakin ingin menghapus semua log files? Tindakan ini tidak dapat dibatalkan!')) {
            // Clear each log file
            const logFiles = " . json_encode(array_column($logFiles, 'name')) . ";
            let cleared = 0;
            
            logFiles.forEach(logFile => {
                const form = document.createElement('form');
                form.method = 'POST';
                form.innerHTML = `
                    <input type=\"hidden\" name=\"action\" value=\"clear_log\">
                    <input type=\"hidden\" name=\"log_file\" value=\"${logFile}\">
                `;
                document.body.appendChild(form);
                
                if (cleared === 0) {
                    form.submit();
                }
                cleared++;
            });
        }
    }
    
    function toggleWrap() {
        const logContent = document.getElementById('logContent');
        if (logContent) {
            if (logContent.style.whiteSpace === 'pre') {
                logContent.style.whiteSpace = 'pre-wrap';
            } else {
                logContent.style.whiteSpace = 'pre';
            }
        }
    }
    
    // Auto-scroll to bottom of log
    document.addEventListener('DOMContentLoaded', function() {
        const logContent = document.getElementById('logContent');
        if (logContent) {
            logContent.scrollTop = logContent.scrollHeight;
        }
    });
    
    // Auto-refresh log content every 10 seconds
    setInterval(() => {
        if (document.visibilityState === 'visible' && window.location.search.includes('log=')) {
            location.reload();
        }
    }, 10000);
";

include 'includes/footer.php';
?>