<?php
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once '../config/database.php';
require_once 'config/admin.php';

AdminAuth::requireLogin();

$pageTitle = 'Dashboard';
$db = new Database();
$conn = $db->getConnection();

// Get statistics
$stats = [];

// Total users
$stmt = $conn->query("SELECT COUNT(*) as total FROM users");
$stats['total_users'] = $stmt->fetch()['total'];

// Total verifications
$stmt = $conn->query("SELECT COUNT(*) as total FROM verifications");
$stats['total_verifications'] = $stmt->fetch()['total'];

// Success verifications
$stmt = $conn->query("SELECT COUNT(*) as total FROM verifications WHERE status = 'success'");
$stats['success_verifications'] = $stmt->fetch()['total'];

// Pending verifications
$stmt = $conn->query("SELECT COUNT(*) as total FROM verifications WHERE status = 'pending'");
$stats['pending_verifications'] = $stmt->fetch()['total'];

// Recent users (last 7 days)
$stmt = $conn->query("SELECT COUNT(*) as total FROM users WHERE created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)");
$stats['recent_users'] = $stmt->fetch()['total'];

// Recent verifications (today)
$stmt = $conn->query("SELECT COUNT(*) as total FROM verifications WHERE DATE(created_at) = CURDATE()");
$stats['today_verifications'] = $stmt->fetch()['total'];

// Get recent activities
$stmt = $conn->query("
    SELECT v.*, u.first_name, u.username 
    FROM verifications v 
    JOIN users u ON v.user_id = u.id 
    ORDER BY v.created_at DESC 
    LIMIT 10
");
$recent_verifications = $stmt->fetchAll();

// Get chart data for last 7 days
$stmt = $conn->query("
    SELECT DATE(created_at) as date, COUNT(*) as count 
    FROM verifications 
    WHERE created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)
    GROUP BY DATE(created_at)
    ORDER BY date
");
$chart_data = $stmt->fetchAll();

// Prepare chart data
$chart_labels = [];
$chart_values = [];
for ($i = 6; $i >= 0; $i--) {
    $date = date('Y-m-d', strtotime("-$i days"));
    $chart_labels[] = date('d/m', strtotime($date));
    
    $found = false;
    foreach ($chart_data as $data) {
        if ($data['date'] == $date) {
            $chart_values[] = $data['count'];
            $found = true;
            break;
        }
    }
    if (!$found) {
        $chart_values[] = 0;
    }
}

include 'includes/header.php';
?>

<div class="row">
    <!-- Statistics Cards -->
    <div class="col-xl-3 col-md-6 mb-4">
        <div class="card stat-card">
            <div class="card-body">
                <div class="d-flex justify-content-between">
                    <div>
                        <h6 class="card-title text-white-50">Total Users</h6>
                        <h2 class="text-white"><?php echo formatNumber($stats['total_users']); ?></h2>
                        <small class="text-white-50">
                            <i class="fas fa-plus me-1"></i>
                            <?php echo $stats['recent_users']; ?> minggu ini
                        </small>
                    </div>
                    <div class="align-self-center">
                        <i class="fas fa-users fa-2x text-white-50"></i>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <div class="col-xl-3 col-md-6 mb-4">
        <div class="card stat-card-success">
            <div class="card-body">
                <div class="d-flex justify-content-between">
                    <div>
                        <h6 class="card-title text-white-50">Verifikasi Sukses</h6>
                        <h2 class="text-white"><?php echo formatNumber($stats['success_verifications']); ?></h2>
                        <small class="text-white-50">
                            <?php 
                            $success_rate = $stats['total_verifications'] > 0 ? 
                                round(($stats['success_verifications'] / $stats['total_verifications']) * 100, 1) : 0;
                            echo $success_rate . '% success rate';
                            ?>
                        </small>
                    </div>
                    <div class="align-self-center">
                        <i class="fas fa-check-circle fa-2x text-white-50"></i>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <div class="col-xl-3 col-md-6 mb-4">
        <div class="card stat-card-warning">
            <div class="card-body">
                <div class="d-flex justify-content-between">
                    <div>
                        <h6 class="card-title text-white-50">Pending</h6>
                        <h2 class="text-white"><?php echo formatNumber($stats['pending_verifications']); ?></h2>
                        <small class="text-white-50">
                            Menunggu proses
                        </small>
                    </div>
                    <div class="align-self-center">
                        <i class="fas fa-clock fa-2x text-white-50"></i>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <div class="col-xl-3 col-md-6 mb-4">
        <div class="card stat-card-info">
            <div class="card-body">
                <div class="d-flex justify-content-between">
                    <div>
                        <h6 class="card-title text-white-50">Hari Ini</h6>
                        <h2 class="text-white"><?php echo formatNumber($stats['today_verifications']); ?></h2>
                        <small class="text-white-50">
                            Verifikasi hari ini
                        </small>
                    </div>
                    <div class="align-self-center">
                        <i class="fas fa-calendar-day fa-2x text-white-50"></i>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<div class="row">
    <!-- Chart -->
    <div class="col-lg-8 mb-4">
        <div class="card">
            <div class="card-header">
                <h5 class="card-title mb-0">
                    <i class="fas fa-chart-line me-2"></i>
                    Verifikasi 7 Hari Terakhir
                </h5>
            </div>
            <div class="card-body">
                <canvas id="verificationsChart" height="100"></canvas>
            </div>
        </div>
    </div>
    
    <!-- Quick Actions -->
    <div class="col-lg-4 mb-4">
        <div class="card">
            <div class="card-header">
                <h5 class="card-title mb-0">
                    <i class="fas fa-bolt me-2"></i>
                    Quick Actions
                </h5>
            </div>
            <div class="card-body">
                <div class="d-grid gap-2">
                    <a href="users.php" class="btn btn-outline-primary">
                        <i class="fas fa-users me-2"></i>Kelola Users
                    </a>
                    <a href="verifications.php" class="btn btn-outline-success">
                        <i class="fas fa-check-circle me-2"></i>Lihat Verifikasi
                    </a>
                    <a href="settings.php" class="btn btn-outline-info">
                        <i class="fas fa-cog me-2"></i>Pengaturan Bot
                    </a>
                    <a href="logs.php" class="btn btn-outline-warning">
                        <i class="fas fa-file-alt me-2"></i>Lihat Logs
                    </a>
                </div>
            </div>
        </div>
        
        <!-- System Status -->
        <div class="card mt-3">
            <div class="card-header">
                <h5 class="card-title mb-0">
                    <i class="fas fa-server me-2"></i>
                    System Status
                </h5>
            </div>
            <div class="card-body">
                <div class="d-flex justify-content-between align-items-center mb-2">
                    <span>Database</span>
                    <span class="badge bg-success">Online</span>
                </div>
                <div class="d-flex justify-content-between align-items-center mb-2">
                    <span>Bot API</span>
                    <span class="badge bg-success">Connected</span>
                </div>
                <div class="d-flex justify-content-between align-items-center">
                    <span>Webhook</span>
                    <span class="badge bg-success">Active</span>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Recent Activities -->
<div class="row">
    <div class="col-12">
        <div class="card">
            <div class="card-header d-flex justify-content-between align-items-center">
                <h5 class="card-title mb-0">
                    <i class="fas fa-history me-2"></i>
                    Aktivitas Terbaru
                </h5>
                <a href="verifications.php" class="btn btn-sm btn-outline-primary">
                    Lihat Semua
                </a>
            </div>
            <div class="card-body">
                <?php if (empty($recent_verifications)): ?>
                    <div class="text-center py-4">
                        <i class="fas fa-inbox fa-3x text-muted mb-3"></i>
                        <p class="text-muted">Belum ada aktivitas verifikasi</p>
                    </div>
                <?php else: ?>
                    <div class="table-responsive">
                        <table class="table table-hover">
                            <thead>
                                <tr>
                                    <th>User</th>
                                    <th>Verification ID</th>
                                    <th>Status</th>
                                    <th>Waktu</th>
                                    <th>Pesan</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($recent_verifications as $verification): ?>
                                    <tr>
                                        <td>
                                            <div class="d-flex align-items-center">
                                                <div class="avatar-sm bg-primary rounded-circle d-flex align-items-center justify-content-center me-2">
                                                    <i class="fas fa-user text-white"></i>
                                                </div>
                                                <div>
                                                    <div class="fw-bold"><?php echo htmlspecialchars($verification['first_name'] ?? 'Unknown'); ?></div>
                                                    <small class="text-muted">@<?php echo htmlspecialchars($verification['username'] ?? 'N/A'); ?></small>
                                                </div>
                                            </div>
                                        </td>
                                        <td>
                                            <code><?php echo htmlspecialchars($verification['verification_id']); ?></code>
                                        </td>
                                        <td><?php echo getStatusBadge($verification['status']); ?></td>
                                        <td>
                                            <small><?php echo formatDate($verification['created_at']); ?></small>
                                        </td>
                                        <td>
                                            <small class="text-muted">
                                                <?php echo htmlspecialchars(substr($verification['message'] ?? '', 0, 50)); ?>
                                                <?php if (strlen($verification['message'] ?? '') > 50) echo '...'; ?>
                                            </small>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<?php
$customJS = "
    // Chart configuration
    const ctx = document.getElementById('verificationsChart').getContext('2d');
    const chart = new Chart(ctx, {
        type: 'line',
        data: {
            labels: " . json_encode($chart_labels) . ",
            datasets: [{
                label: 'Verifikasi',
                data: " . json_encode($chart_values) . ",
                borderColor: 'rgb(102, 126, 234)',
                backgroundColor: 'rgba(102, 126, 234, 0.1)',
                borderWidth: 3,
                fill: true,
                tension: 0.4
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    display: false
                }
            },
            scales: {
                y: {
                    beginAtZero: true,
                    ticks: {
                        stepSize: 1
                    }
                }
            }
        }
    });
";

include 'includes/footer.php';
?>