<?php
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once '../config/database.php';
require_once 'config/admin.php';

AdminAuth::requireLogin();

require_once '../models/GlobalSettings.php';
$globalSettings = new GlobalSettings();

$pageTitle = 'Commands & Messages';
$success = '';
$error = '';
$error = '';

// Handle form submissions
if ($_POST) {
    try {
        if (isset($_POST['update_commands'])) {
            // Update command settings
            $commandSettings = [
                'start_message' => $_POST['start_message'] ?? '',
                'help_message' => $_POST['help_message'] ?? '',
                'verify_success_message' => $_POST['verify_success_message'] ?? '',
                'verify_error_message' => $_POST['verify_error_message'] ?? '',
                'batch_success_message' => $_POST['batch_success_message'] ?? '',
                'batch_error_message' => $_POST['batch_error_message'] ?? '',
                'status_message' => $_POST['status_message'] ?? '',
                'maintenance_message' => $_POST['maintenance_message'] ?? '',
                'access_denied_message' => $_POST['access_denied_message'] ?? '',
                'unknown_command_message' => $_POST['unknown_command_message'] ?? ''
            ];
            
            $updateCount = 0;
            $errors = [];
            
            foreach ($commandSettings as $key => $value) {
                $result = $globalSettings->setSetting($key, $value, "Custom message for $key");
                if ($result) {
                    $updateCount++;
                } else {
                    $errors[] = $key;
                }
            }
            
            if ($updateCount > 0) {
                $success = "✅ Command messages updated successfully! ($updateCount settings updated)";
                if (!empty($errors)) {
                    $success .= " (Failed: " . implode(', ', $errors) . ")";
                }
            } else {
                $error = "❌ Failed to update settings. Check database connection. Failed keys: " . implode(', ', $errors);
            }
        }
        
        if (isset($_POST['update_menu'])) {
            // Update menu settings
            $menuSettings = [
                'show_help_button' => isset($_POST['show_help_button']) ? '1' : '0',
                'show_status_button' => isset($_POST['show_status_button']) ? '1' : '0',
                'show_verify_button' => isset($_POST['show_verify_button']) ? '1' : '0',
                'show_batch_button' => isset($_POST['show_batch_button']) ? '1' : '0',
                'show_history_button' => isset($_POST['show_history_button']) ? '1' : '0',
                'show_admin_button' => isset($_POST['show_admin_button']) ? '1' : '0',
                'show_web_admin_link' => isset($_POST['show_web_admin_link']) ? '1' : '0',
                'custom_menu_buttons' => $_POST['custom_menu_buttons'] ?? ''
            ];
            
            $updateCount = 0;
            foreach ($menuSettings as $key => $value) {
                if ($globalSettings->setSetting($key, $value, "Menu setting for $key")) {
                    $updateCount++;
                }
            }
            
            if ($updateCount > 0) {
                $success = "✅ Menu settings updated successfully! ($updateCount settings updated)";
            } else {
                $error = "❌ Failed to update menu settings. Check database connection.";
            }
        }
        
        if (isset($_POST['reset_defaults'])) {
            // Reset to default messages
            $defaultMessages = [
                'start_message' => "🤖 <b>Selamat datang di Verification Bot!</b>\n\nBot ini membantu Anda melakukan verifikasi menggunakan API.",
                'help_message' => "📖 <b>Bantuan Verification Bot</b>\n\nGunakan command berikut untuk berinteraksi dengan bot:",
                'verify_success_message' => "✅ <b>Verifikasi dimulai!</b>\n\nProses sedang berjalan... Anda akan mendapat notifikasi saat selesai.",
                'verify_error_message' => "❌ <b>Gagal memulai verifikasi!</b>",
                'batch_success_message' => "✅ <b>Batch verifikasi dimulai!</b>\n\nProses sedang berjalan... Anda akan mendapat notifikasi saat selesai.",
                'batch_error_message' => "❌ <b>Gagal memulai batch verifikasi!</b>",
                'status_message' => "🔍 <b>Status Sistem</b>",
                'maintenance_message' => "🔧 <b>Bot sedang dalam maintenance</b>\n\nSilakan coba lagi nanti.",
                'access_denied_message' => "❌ <b>Akses Ditolak!</b>\n\nAnda tidak memiliki izin untuk menggunakan command ini.",
                'unknown_command_message' => "❓ <b>Perintah tidak dikenal!</b>\n\nGunakan /help untuk melihat daftar perintah yang tersedia."
            ];
            
            $updateCount = 0;
            foreach ($defaultMessages as $key => $value) {
                if ($globalSettings->setSetting($key, $value, "Default message for $key")) {
                    $updateCount++;
                }
            }
            
            $success = "✅ Messages reset to defaults successfully! ($updateCount messages reset)";
        }
        
    } catch (Exception $e) {
        $error = "Error: " . $e->getMessage();
    }
}

// Get current settings
$currentMessages = [
    'start_message' => $globalSettings->getSetting('start_message', '🤖 <b>Selamat datang di Verification Bot!</b>\n\nBot ini membantu Anda melakukan verifikasi menggunakan API.'),
    'help_message' => $globalSettings->getSetting('help_message', '📖 <b>Bantuan Verification Bot</b>\n\nGunakan command berikut untuk berinteraksi dengan bot:'),
    'verify_success_message' => $globalSettings->getSetting('verify_success_message', '✅ <b>Verifikasi dimulai!</b>\n\nProses sedang berjalan... Anda akan mendapat notifikasi saat selesai.'),
    'verify_error_message' => $globalSettings->getSetting('verify_error_message', '❌ <b>Gagal memulai verifikasi!</b>'),
    'batch_success_message' => $globalSettings->getSetting('batch_success_message', '✅ <b>Batch verifikasi dimulai!</b>\n\nProses sedang berjalan... Anda akan mendapat notifikasi saat selesai.'),
    'batch_error_message' => $globalSettings->getSetting('batch_error_message', '❌ <b>Gagal memulai batch verifikasi!</b>'),
    'status_message' => $globalSettings->getSetting('status_message', '🔍 <b>Status Sistem</b>'),
    'maintenance_message' => $globalSettings->getSetting('maintenance_message', '🔧 <b>Bot sedang dalam maintenance</b>\n\nSilakan coba lagi nanti.'),
    'access_denied_message' => $globalSettings->getSetting('access_denied_message', '❌ <b>Akses Ditolak!</b>\n\nAnda tidak memiliki izin untuk menggunakan command ini.'),
    'unknown_command_message' => $globalSettings->getSetting('unknown_command_message', '❓ <b>Perintah tidak dikenal!</b>\n\nGunakan /help untuk melihat daftar perintah yang tersedia.')
];

$currentMenu = [
    'show_help_button' => $globalSettings->getSetting('show_help_button', '1'),
    'show_admin_button' => $globalSettings->getSetting('show_admin_button', '1'),
    'show_web_admin_link' => $globalSettings->getSetting('show_web_admin_link', '1'),
    'custom_menu_buttons' => $globalSettings->getSetting('custom_menu_buttons', '')
];

include 'includes/header.php';
?>

<?php if ($success): ?>
    <div class="alert alert-success alert-dismissible fade show">
        <i class="fas fa-check-circle me-2"></i><?php echo $success; ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
<?php endif; ?>

<?php if ($error): ?>
    <div class="alert alert-danger alert-dismissible fade show">
        <i class="fas fa-exclamation-triangle me-2"></i><?php echo $error; ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
<?php endif; ?>

<div class="row">
    <!-- Command Messages -->
    <div class="col-lg-8 mb-4">
        <div class="card">
            <div class="card-header">
                <h5 class="card-title mb-0">
                    <i class="fas fa-comments me-2"></i>
                    Bot Command Messages
                </h5>
            </div>
            <div class="card-body">
                <form method="POST">
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">
                                <i class="fas fa-play-circle text-success me-1"></i>
                                Start Message (/start)
                            </label>
                            <textarea class="form-control" name="start_message" rows="3" placeholder="Welcome message for /start command"><?php echo htmlspecialchars($currentMessages['start_message']); ?></textarea>
                            <small class="text-muted">Pesan yang ditampilkan saat user mengetik /start</small>
                        </div>
                        
                        <div class="col-md-6 mb-3">
                            <label class="form-label">
                                <i class="fas fa-question-circle text-info me-1"></i>
                                Help Message (/help)
                            </label>
                            <textarea class="form-control" name="help_message" rows="3" placeholder="Help message for /help command"><?php echo htmlspecialchars($currentMessages['help_message']); ?></textarea>
                            <small class="text-muted">Pesan yang ditampilkan saat user mengetik /help</small>
                        </div>
                        
                        <div class="col-md-6 mb-3">
                            <label class="form-label">
                                <i class="fas fa-check-circle text-success me-1"></i>
                                Verify Success Message
                            </label>
                            <textarea class="form-control" name="verify_success_message" rows="3" placeholder="Success message for verification"><?php echo htmlspecialchars($currentMessages['verify_success_message']); ?></textarea>
                            <small class="text-muted">Pesan saat verifikasi berhasil dimulai</small>
                        </div>
                        
                        <div class="col-md-6 mb-3">
                            <label class="form-label">
                                <i class="fas fa-times-circle text-danger me-1"></i>
                                Verify Error Message
                            </label>
                            <textarea class="form-control" name="verify_error_message" rows="3" placeholder="Error message for verification"><?php echo htmlspecialchars($currentMessages['verify_error_message']); ?></textarea>
                            <small class="text-muted">Pesan saat verifikasi gagal</small>
                        </div>
                        
                        <div class="col-md-6 mb-3">
                            <label class="form-label">
                                <i class="fas fa-layer-group text-primary me-1"></i>
                                Batch Success Message
                            </label>
                            <textarea class="form-control" name="batch_success_message" rows="3" placeholder="Success message for batch verification"><?php echo htmlspecialchars($currentMessages['batch_success_message']); ?></textarea>
                            <small class="text-muted">Pesan saat batch verifikasi berhasil dimulai</small>
                        </div>
                        
                        <div class="col-md-6 mb-3">
                            <label class="form-label">
                                <i class="fas fa-exclamation-triangle text-warning me-1"></i>
                                Batch Error Message
                            </label>
                            <textarea class="form-control" name="batch_error_message" rows="3" placeholder="Error message for batch verification"><?php echo htmlspecialchars($currentMessages['batch_error_message']); ?></textarea>
                            <small class="text-muted">Pesan saat batch verifikasi gagal</small>
                        </div>
                        
                        <div class="col-md-6 mb-3">
                            <label class="form-label">
                                <i class="fas fa-info-circle text-info me-1"></i>
                                Status Message
                            </label>
                            <textarea class="form-control" name="status_message" rows="2" placeholder="Header for status command"><?php echo htmlspecialchars($currentMessages['status_message']); ?></textarea>
                            <small class="text-muted">Header untuk command /status</small>
                        </div>
                        
                        <div class="col-md-6 mb-3">
                            <label class="form-label">
                                <i class="fas fa-tools text-warning me-1"></i>
                                Maintenance Message
                            </label>
                            <textarea class="form-control" name="maintenance_message" rows="2" placeholder="Message when bot is in maintenance"><?php echo htmlspecialchars($currentMessages['maintenance_message']); ?></textarea>
                            <small class="text-muted">Pesan saat bot dalam maintenance</small>
                        </div>
                        
                        <div class="col-md-6 mb-3">
                            <label class="form-label">
                                <i class="fas fa-ban text-danger me-1"></i>
                                Access Denied Message
                            </label>
                            <textarea class="form-control" name="access_denied_message" rows="2" placeholder="Message for access denied"><?php echo htmlspecialchars($currentMessages['access_denied_message']); ?></textarea>
                            <small class="text-muted">Pesan saat akses ditolak</small>
                        </div>
                        
                        <div class="col-md-6 mb-3">
                            <label class="form-label">
                                <i class="fas fa-question text-secondary me-1"></i>
                                Unknown Command Message
                            </label>
                            <textarea class="form-control" name="unknown_command_message" rows="2" placeholder="Message for unknown commands"><?php echo htmlspecialchars($currentMessages['unknown_command_message']); ?></textarea>
                            <small class="text-muted">Pesan saat command tidak dikenal</small>
                        </div>
                    </div>
                    
                    <div class="d-flex gap-2">
                        <button type="submit" name="update_commands" class="btn btn-primary">
                            <i class="fas fa-save me-1"></i>Update Messages
                        </button>
                        <button type="submit" name="reset_defaults" class="btn btn-outline-secondary" onclick="return confirm('Reset all messages to defaults?')">
                            <i class="fas fa-undo me-1"></i>Reset to Defaults
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <!-- Menu Settings -->
    <div class="col-lg-4 mb-4">
        <div class="card">
            <div class="card-header">
                <h5 class="card-title mb-0">
                    <i class="fas fa-bars me-2"></i>
                    Menu Settings
                </h5>
            </div>
            <div class="card-body">
                <form method="POST">
                    <div class="mb-3">
                        <label class="form-label">Menu Button Configuration</label>
                        
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" name="show_help_button" id="show_help_button" <?php echo $currentMenu['show_help_button'] ? 'checked' : ''; ?>>
                            <label class="form-check-label" for="show_help_button">
                                <i class="fas fa-question-circle text-info me-1"></i>Show Help & Panduan Button
                            </label>
                        </div>
                        
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" name="show_status_button" id="show_status_button" <?php echo $globalSettings->getSetting('show_status_button', '1') ? 'checked' : ''; ?>>
                            <label class="form-check-label" for="show_status_button">
                                <i class="fas fa-chart-line text-primary me-1"></i>Show Status Sistem Button
                            </label>
                        </div>
                        
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" name="show_verify_button" id="show_verify_button" <?php echo $globalSettings->getSetting('show_verify_button', '1') ? 'checked' : ''; ?>>
                            <label class="form-check-label" for="show_verify_button">
                                <i class="fas fa-check-circle text-success me-1"></i>Show Verifikasi Single Button
                            </label>
                        </div>
                        
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" name="show_batch_button" id="show_batch_button" <?php echo $globalSettings->getSetting('show_batch_button', '1') ? 'checked' : ''; ?>>
                            <label class="form-check-label" for="show_batch_button">
                                <i class="fas fa-layer-group text-warning me-1"></i>Show Verifikasi Batch Button
                            </label>
                        </div>
                        
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" name="show_history_button" id="show_history_button" <?php echo $globalSettings->getSetting('show_history_button', '1') ? 'checked' : ''; ?>>
                            <label class="form-check-label" for="show_history_button">
                                <i class="fas fa-history text-secondary me-1"></i>Show Riwayat Verifikasi Button
                            </label>
                        </div>
                        
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" name="show_admin_button" id="show_admin_button" <?php echo $currentMenu['show_admin_button'] ? 'checked' : ''; ?>>
                            <label class="form-check-label" for="show_admin_button">
                                <i class="fas fa-crown text-warning me-1"></i>Show Admin Panel Button (Admin Only)
                            </label>
                        </div>
                        
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" name="show_web_admin_link" id="show_web_admin_link" <?php echo $currentMenu['show_web_admin_link'] ? 'checked' : ''; ?>>
                            <label class="form-check-label" for="show_web_admin_link">
                                <i class="fas fa-external-link-alt text-primary me-1"></i>Show Web Admin Link (Admin Only)
                            </label>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Custom Menu Buttons (JSON)</label>
                        <textarea class="form-control" name="custom_menu_buttons" rows="4" placeholder='[{"text": "Custom Button", "callback_data": "custom_action"}]'><?php echo htmlspecialchars($currentMenu['custom_menu_buttons']); ?></textarea>
                        <small class="text-muted">Format JSON untuk button tambahan (opsional)</small>
                    </div>
                    
                    <button type="submit" name="update_menu" class="btn btn-success">
                        <i class="fas fa-save me-1"></i>Update Menu
                    </button>
                </form>
            </div>
        </div>
        
        <!-- Preview -->
        <div class="card mt-3">
            <div class="card-header">
                <h6 class="card-title mb-0">
                    <i class="fas fa-eye me-2"></i>
                    Message Preview
                </h6>
            </div>
            <div class="card-body">
                <div class="mb-2">
                    <strong>Start Message:</strong>
                    <div class="border p-2 mt-1 small bg-light">
                        <?php echo nl2br(htmlspecialchars(substr($currentMessages['start_message'], 0, 100))); ?>
                        <?php if (strlen($currentMessages['start_message']) > 100) echo '...'; ?>
                    </div>
                </div>
                
                <div class="mb-2">
                    <strong>Help Message:</strong>
                    <div class="border p-2 mt-1 small bg-light">
                        <?php echo nl2br(htmlspecialchars(substr($currentMessages['help_message'], 0, 100))); ?>
                        <?php if (strlen($currentMessages['help_message']) > 100) echo '...'; ?>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Command List -->
        <div class="card mt-3">
            <div class="card-header">
                <h6 class="card-title mb-0">
                    <i class="fas fa-list me-2"></i>
                    Available Commands
                </h6>
                <div class="command-stats"></div>
            </div>
            <div class="card-body command-list-detail">
                <div class="text-center">
                    <div class="spinner-border spinner-border-sm" role="status">
                        <span class="visually-hidden">Loading...</span>
                    </div>
                    <div class="small text-muted mt-2">Loading command info...</div>
                </div>
            </div>
        </div>
        
        <!-- Message Preview -->
        <div class="card mt-3">
            <div class="card-header">
                <h6 class="card-title mb-0">
                    <i class="fas fa-eye me-2"></i>
                    Message Preview
                </h6>
            </div>
            <div class="card-body message-preview-content">
                <div class="text-muted text-center">
                    <i class="fas fa-info-circle"></i>
                    <div class="small mt-2">Click preview button on any message to see how it will look</div>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
// Auto-resize textareas
document.querySelectorAll('textarea').forEach(textarea => {
    textarea.addEventListener('input', function() {
        this.style.height = 'auto';
        this.style.height = this.scrollHeight + 'px';
    });
});

// Load command information
async function loadCommandInfo() {
    try {
        const response = await fetch('ajax/get_command_info.php');
        const data = await response.json();
        
        // Update command list in sidebar
        updateCommandList(data.commands, data.system_messages);
        
        // Update stats
        document.querySelector('.command-stats').innerHTML = `
            <small class="text-muted">
                Total: ${data.total_commands} commands | 
                User: ${data.user_commands} | 
                Admin: ${data.admin_commands}
            </small>
        `;
        
    } catch (error) {
        console.error('Error loading command info:', error);
    }
}

function updateCommandList(commands, systemMessages) {
    const commandListElement = document.querySelector('.command-list-detail');
    if (!commandListElement) return;
    
    let html = '<div class="small">';
    
    // User commands
    html += '<h6 class="text-primary mb-2">User Commands:</h6>';
    commands.filter(cmd => !cmd.admin_only).forEach(cmd => {
        html += `
            <div class="mb-2 p-2 border rounded">
                <div class="fw-bold">${cmd.command}</div>
                <div class="text-muted">${cmd.description}</div>
                <div><code>${cmd.usage}</code></div>
                ${cmd.example ? `<div class="text-success small">Example: <code>${cmd.example}</code></div>` : ''}
            </div>
        `;
    });
    
    // Admin commands
    html += '<h6 class="text-warning mb-2 mt-3">Admin Commands:</h6>';
    commands.filter(cmd => cmd.admin_only).forEach(cmd => {
        html += `
            <div class="mb-2 p-2 border rounded bg-light">
                <div class="fw-bold">${cmd.command} <span class="badge bg-warning text-dark">Admin</span></div>
                <div class="text-muted">${cmd.description}</div>
                <div><code>${cmd.usage}</code></div>
                ${cmd.example ? `<div class="text-success small">Example: <code>${cmd.example}</code></div>` : ''}
            </div>
        `;
    });
    
    // System messages
    html += '<h6 class="text-info mb-2 mt-3">System Messages:</h6>';
    systemMessages.forEach(msg => {
        html += `
            <div class="mb-2 p-2 border rounded bg-info bg-opacity-10">
                <div class="fw-bold">${msg.name}</div>
                <div class="text-muted small">${msg.description}</div>
            </div>
        `;
    });
    
    html += '</div>';
    commandListElement.innerHTML = html;
}

// Load on page ready
document.addEventListener('DOMContentLoaded', loadCommandInfo);

// Preview message function
function previewMessage(messageType) {
    const textarea = document.querySelector(`textarea[name="${messageType}"]`);
    if (!textarea) return;
    
    const message = textarea.value;
    const previewDiv = document.querySelector('.message-preview-content');
    
    if (previewDiv) {
        previewDiv.innerHTML = `
            <h6>${messageType.replace('_', ' ').toUpperCase()}</h6>
            <div class="border p-3 bg-light rounded">
                ${message.replace(/\n/g, '<br>').replace(/\*\*(.*?)\*\*/g, '<strong>$1</strong>')}
            </div>
        `;
    }
}

// Add preview buttons to textareas
document.querySelectorAll('textarea').forEach(textarea => {
    const messageType = textarea.name;
    const previewBtn = document.createElement('button');
    previewBtn.type = 'button';
    previewBtn.className = 'btn btn-sm btn-outline-info mt-1';
    previewBtn.innerHTML = '<i class="fas fa-eye"></i> Preview';
    previewBtn.onclick = () => previewMessage(messageType);
    
    textarea.parentNode.appendChild(previewBtn);
});
</script>

<?php include 'includes/footer.php'; ?>