<?php
require_once '../config/database.php';
require_once 'config/admin.php';

AdminAuth::requireLogin();

$pageTitle = 'Batch Verifications';
$db = new Database();
$conn = $db->getConnection();

// Handle actions
if ($_POST) {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'update_batch_status':
                $batchId = $_POST['batch_id'];
                $status = $_POST['status'];
                
                $stmt = $conn->prepare("UPDATE batch_verifications SET status = ?, updated_at = NOW() WHERE batch_id = ?");
                if ($stmt->execute([$status, $batchId])) {
                    $success = "Status batch berhasil diperbarui!";
                } else {
                    $error = "Gagal memperbarui status batch!";
                }
                break;
                
            case 'delete_batch':
                $batchId = $_POST['batch_id'];
                $stmt = $conn->prepare("DELETE FROM batch_verifications WHERE batch_id = ?");
                if ($stmt->execute([$batchId])) {
                    $success = "Batch berhasil dihapus!";
                } else {
                    $error = "Gagal menghapus batch!";
                }
                break;
        }
    }
}

// Filters
$status = $_GET['status'] ?? '';
$search = $_GET['search'] ?? '';
$dateFrom = $_GET['date_from'] ?? '';
$dateTo = $_GET['date_to'] ?? '';

// Build WHERE clause
$whereConditions = [];
$params = [];

if ($status) {
    $whereConditions[] = "bv.status = ?";
    $params[] = $status;
}

if ($search) {
    $whereConditions[] = "(bv.batch_id LIKE ? OR u.first_name LIKE ? OR u.username LIKE ?)";
    $params[] = "%$search%";
    $params[] = "%$search%";
    $params[] = "%$search%";
}

if ($dateFrom) {
    $whereConditions[] = "DATE(bv.created_at) >= ?";
    $params[] = $dateFrom;
}

if ($dateTo) {
    $whereConditions[] = "DATE(bv.created_at) <= ?";
    $params[] = $dateTo;
}

$whereClause = !empty($whereConditions) ? 'WHERE ' . implode(' AND ', $whereConditions) : '';

// Pagination
$page = $_GET['page'] ?? 1;
$limit = ITEMS_PER_PAGE;
$offset = ($page - 1) * $limit;

// Get total count
$countQuery = "SELECT COUNT(*) as total FROM batch_verifications bv JOIN users u ON bv.user_id = u.id $whereClause";
$stmt = $conn->prepare($countQuery);
$stmt->execute($params);
$totalBatches = $stmt->fetch()['total'];
$totalPages = ceil($totalBatches / $limit);

// Get batch verifications
$query = "
    SELECT bv.*, u.first_name, u.last_name, u.username, u.telegram_id
    FROM batch_verifications bv 
    JOIN users u ON bv.user_id = u.id 
    $whereClause
    ORDER BY bv.created_at DESC 
    LIMIT $limit OFFSET $offset
";

$stmt = $conn->prepare($query);
$stmt->execute($params);
$batches = $stmt->fetchAll();

// Get status counts
$statusCounts = [];
$statusQuery = "SELECT status, COUNT(*) as count FROM batch_verifications GROUP BY status";
$stmt = $conn->query($statusQuery);
while ($row = $stmt->fetch()) {
    $statusCounts[$row['status']] = $row['count'];
}

include 'includes/header.php';
?>

<!-- Filters -->
<div class="card mb-4">
    <div class="card-body">
        <form method="GET" class="row g-3">
            <div class="col-md-3">
                <label class="form-label">Cari</label>
                <input type="text" class="form-control" name="search" 
                       value="<?php echo htmlspecialchars($search); ?>" 
                       placeholder="Batch ID, nama, atau username">
            </div>
            
            <div class="col-md-2">
                <label class="form-label">Status</label>
                <select class="form-select" name="status">
                    <option value="">Semua Status</option>
                    <option value="pending" <?php echo $status === 'pending' ? 'selected' : ''; ?>>Pending</option>
                    <option value="processing" <?php echo $status === 'processing' ? 'selected' : ''; ?>>Processing</option>
                    <option value="completed" <?php echo $status === 'completed' ? 'selected' : ''; ?>>Completed</option>
                    <option value="error" <?php echo $status === 'error' ? 'selected' : ''; ?>>Error</option>
                </select>
            </div>
            
            <div class="col-md-2">
                <label class="form-label">Dari Tanggal</label>
                <input type="date" class="form-control" name="date_from" 
                       value="<?php echo htmlspecialchars($dateFrom); ?>">
            </div>
            
            <div class="col-md-2">
                <label class="form-label">Sampai Tanggal</label>
                <input type="date" class="form-control" name="date_to" 
                       value="<?php echo htmlspecialchars($dateTo); ?>">
            </div>
            
            <div class="col-md-3">
                <label class="form-label">&nbsp;</label>
                <div class="d-flex gap-2">
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-search me-1"></i>Filter
                    </button>
                    <a href="batches.php" class="btn btn-outline-secondary">
                        <i class="fas fa-times me-1"></i>Reset
                    </a>
                </div>
            </div>
        </form>
    </div>
</div>

<?php if (isset($success)): ?>
    <div class="alert alert-success alert-dismissible fade show">
        <i class="fas fa-check-circle me-2"></i><?php echo $success; ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
<?php endif; ?>

<?php if (isset($error)): ?>
    <div class="alert alert-danger alert-dismissible fade show">
        <i class="fas fa-exclamation-triangle me-2"></i><?php echo $error; ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
<?php endif; ?>

<!-- Status Summary -->
<div class="row mb-4">
    <div class="col-12">
        <div class="btn-group w-100" role="group">
            <a href="batches.php" class="btn <?php echo !$status ? 'btn-primary' : 'btn-outline-primary'; ?>">
                Semua (<?php echo formatNumber($totalBatches); ?>)
            </a>
            <a href="?status=pending" class="btn <?php echo $status === 'pending' ? 'btn-warning' : 'btn-outline-warning'; ?>">
                Pending (<?php echo formatNumber($statusCounts['pending'] ?? 0); ?>)
            </a>
            <a href="?status=processing" class="btn <?php echo $status === 'processing' ? 'btn-info' : 'btn-outline-info'; ?>">
                Processing (<?php echo formatNumber($statusCounts['processing'] ?? 0); ?>)
            </a>
            <a href="?status=completed" class="btn <?php echo $status === 'completed' ? 'btn-success' : 'btn-outline-success'; ?>">
                Completed (<?php echo formatNumber($statusCounts['completed'] ?? 0); ?>)
            </a>
            <a href="?status=error" class="btn <?php echo $status === 'error' ? 'btn-danger' : 'btn-outline-danger'; ?>">
                Error (<?php echo formatNumber($statusCounts['error'] ?? 0); ?>)
            </a>
        </div>
    </div>
</div>

<!-- Batches Table -->
<div class="card">
    <div class="card-header d-flex justify-content-between align-items-center">
        <h5 class="card-title mb-0">
            <i class="fas fa-layer-group me-2"></i>
            Daftar Batch Verifications
        </h5>
        <div class="btn-group">
            <button class="btn btn-outline-primary btn-sm" onclick="location.reload()">
                <i class="fas fa-sync-alt me-1"></i>Refresh
            </button>
        </div>
    </div>
    <div class="card-body">
        <?php if (empty($batches)): ?>
            <div class="text-center py-5">
                <i class="fas fa-layer-group fa-3x text-muted mb-3"></i>
                <h5 class="text-muted">Tidak ada batch ditemukan</h5>
                <p class="text-muted">Coba ubah filter pencarian Anda</p>
            </div>
        <?php else: ?>
            <div class="table-responsive">
                <table class="table table-hover">
                    <thead>
                        <tr>
                            <th>User</th>
                            <th>Batch ID</th>
                            <th>Progress</th>
                            <th>Status</th>
                            <th>Waktu</th>
                            <th>Aksi</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($batches as $batch): ?>
                            <tr>
                                <td>
                                    <div class="d-flex align-items-center">
                                        <div class="avatar-sm bg-primary rounded-circle d-flex align-items-center justify-content-center me-2">
                                            <i class="fas fa-user text-white"></i>
                                        </div>
                                        <div>
                                            <div class="fw-bold">
                                                <?php echo htmlspecialchars($batch['first_name'] ?? 'Unknown'); ?>
                                                <?php if ($batch['last_name']): ?>
                                                    <?php echo htmlspecialchars($batch['last_name']); ?>
                                                <?php endif; ?>
                                            </div>
                                            <small class="text-muted">@<?php echo htmlspecialchars($batch['username'] ?? 'N/A'); ?></small>
                                        </div>
                                    </div>
                                </td>
                                <td>
                                    <code><?php echo htmlspecialchars($batch['batch_id']); ?></code>
                                    <button class="btn btn-sm btn-outline-secondary ms-1" 
                                            onclick="copyToClipboard('<?php echo htmlspecialchars($batch['batch_id']); ?>')"
                                            data-bs-toggle="tooltip" title="Copy ID">
                                        <i class="fas fa-copy"></i>
                                    </button>
                                </td>
                                <td>
                                    <div class="d-flex align-items-center">
                                        <div class="progress flex-grow-1 me-2" style="height: 20px;">
                                            <?php 
                                            $percentage = $batch['total_verifications'] > 0 ? 
                                                ($batch['completed_verifications'] / $batch['total_verifications']) * 100 : 0;
                                            ?>
                                            <div class="progress-bar" role="progressbar" 
                                                 style="width: <?php echo $percentage; ?>%">
                                                <?php echo round($percentage, 1); ?>%
                                            </div>
                                        </div>
                                        <small class="text-muted">
                                            <?php echo $batch['completed_verifications']; ?>/<?php echo $batch['total_verifications']; ?>
                                        </small>
                                    </div>
                                </td>
                                <td>
                                    <span id="batch-status-<?php echo $batch['id']; ?>">
                                        <?php echo getStatusBadge($batch['status']); ?>
                                    </span>
                                </td>
                                <td>
                                    <div>
                                        <small><?php echo formatDate($batch['created_at']); ?></small>
                                        <?php if ($batch['updated_at'] !== $batch['created_at']): ?>
                                            <br><small class="text-muted">Updated: <?php echo formatDate($batch['updated_at']); ?></small>
                                        <?php endif; ?>
                                    </div>
                                </td>
                                <td>
                                    <div class="btn-group btn-group-sm">
                                        <button class="btn btn-outline-info btn-action" 
                                                onclick="viewBatchDetails('<?php echo $batch['batch_id']; ?>')"
                                                data-bs-toggle="tooltip" title="Lihat Detail">
                                            <i class="fas fa-eye"></i>
                                        </button>
                                        
                                        <button class="btn btn-outline-primary btn-action" 
                                                onclick="editBatchStatus('<?php echo $batch['batch_id']; ?>', '<?php echo $batch['status']; ?>')"
                                                data-bs-toggle="tooltip" title="Edit Status">
                                            <i class="fas fa-edit"></i>
                                        </button>
                                        
                                        <button class="btn btn-outline-danger btn-action" 
                                                onclick="deleteBatch('<?php echo $batch['batch_id']; ?>')"
                                                data-bs-toggle="tooltip" title="Hapus">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            
            <!-- Pagination -->
            <?php if ($totalPages > 1): ?>
                <nav class="mt-4">
                    <ul class="pagination justify-content-center">
                        <?php if ($page > 1): ?>
                            <li class="page-item">
                                <a class="page-link" href="?page=<?php echo $page-1; ?>&<?php echo http_build_query($_GET); ?>">
                                    <i class="fas fa-chevron-left"></i>
                                </a>
                            </li>
                        <?php endif; ?>
                        
                        <?php for ($i = max(1, $page-2); $i <= min($totalPages, $page+2); $i++): ?>
                            <li class="page-item <?php echo $i == $page ? 'active' : ''; ?>">
                                <a class="page-link" href="?page=<?php echo $i; ?>&<?php echo http_build_query(array_merge($_GET, ['page' => $i])); ?>">
                                    <?php echo $i; ?>
                                </a>
                            </li>
                        <?php endfor; ?>
                        
                        <?php if ($page < $totalPages): ?>
                            <li class="page-item">
                                <a class="page-link" href="?page=<?php echo $page+1; ?>&<?php echo http_build_query($_GET); ?>">
                                    <i class="fas fa-chevron-right"></i>
                                </a>
                            </li>
                        <?php endif; ?>
                    </ul>
                </nav>
            <?php endif; ?>
        <?php endif; ?>
    </div>
</div>

<!-- Edit Batch Status Modal -->
<div class="modal fade" id="editBatchModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Edit Status Batch</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST">
                <div class="modal-body">
                    <input type="hidden" name="action" value="update_batch_status">
                    <input type="hidden" name="batch_id" id="edit_batch_id">
                    
                    <div class="mb-3">
                        <label class="form-label">Status</label>
                        <select class="form-select" name="status" id="edit_batch_status" required>
                            <option value="pending">Pending</option>
                            <option value="processing">Processing</option>
                            <option value="completed">Completed</option>
                            <option value="error">Error</option>
                        </select>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                    <button type="submit" class="btn btn-primary">Simpan</button>
                </div>
            </form>
        </div>
    </div>
</div>

<?php
$customJS = "
    function editBatchStatus(batchId, status) {
        document.getElementById('edit_batch_id').value = batchId;
        document.getElementById('edit_batch_status').value = status;
        
        const modal = new bootstrap.Modal(document.getElementById('editBatchModal'));
        modal.show();
    }
    
    function deleteBatch(batchId) {
        if (confirm('Apakah Anda yakin ingin menghapus batch ini?')) {
            const form = document.createElement('form');
            form.method = 'POST';
            form.innerHTML = `
                <input type=\"hidden\" name=\"action\" value=\"delete_batch\">
                <input type=\"hidden\" name=\"batch_id\" value=\"${batchId}\">
            `;
            document.body.appendChild(form);
            form.submit();
        }
    }
    
    function viewBatchDetails(batchId) {
        // Redirect to verifications page with batch filter
        window.location.href = 'verifications.php?search=' + encodeURIComponent(batchId);
    }
    
    // Auto-refresh every 30 seconds
    setInterval(() => {
        if (document.visibilityState === 'visible') {
            if (!document.querySelector('.modal.show')) {
                location.reload();
            }
        }
    }, 30000);
";

include 'includes/footer.php';
?>